const BIG_PRECISION = 16

export const prices = {
  BTCUSDT: -2,
  ETHUSDT: -2,
}

export const quantities = {
  BTCUSDT: -3,
  ETHUSDT: -2,
}

export const quotes = {
  BTCUSDT: prices.BTCUSDT + quantities.BTCUSDT,
  ETHUSDT: prices.ETHUSDT + quantities.ETHUSDT,
}

export const precisions = {
  BTCUSDT: {
    price: prices.BTCUSDT,
    quantity: quantities.BTCUSDT,
    quote: quotes.BTCUSDT,
    total: quotes.BTCUSDT,
  },
  ETHUSDT: {
    price: prices.ETHUSDT,
    quantity: quantities.ETHUSDT,
    quote: quotes.ETHUSDT,
    total: quotes.ETHUSDT,
  },
  FISH: {
    margin: -4,
    quantity: -4,
  },
}

const aliases = {
  symbols: {
    BTCfish: 'BTCUSDT',
    ETHfish: 'ETHUSDT',
  },
  types: {
    cost: 'price',
    entry: 'price',
    open: 'quantity',
    close: 'quantity',
    long: 'quantity',
    short: 'quantity',
    notional: 'quote',
    margin: 'quote',
    leverage: 'int',
  },
}

export const fixVal = ({ value, type, symbol, apx }) => {
  if (!precisions[symbol] && aliases.symbols[symbol])
    symbol = aliases.symbols[symbol]

  if (!precisions[symbol][type] && aliases.types[type])
    type = aliases.types[type]

  if (!precisions[symbol])
    throw new Error(`Unknown symbol: ${symbol}`)
  if (!precisions[symbol][type])
    throw new Error(`No precision for ${type} ${symbol}`)

  return fixNumber(value, precisions[symbol][type], apx)
}
export const fixPrice = (price, symbol, apx) => fixVal({ value: price, type: 'price', symbol, apx })
export const fixQty = (qty, symbol, apx) => fixVal({ value: qty, type: 'quantity', symbol, apx })
export const fixQuote = (quote, symbol, apx) => fixVal({ value: quote, type: 'quote', symbol, apx })

// tick 10^P => precision P, 0.001 => -3, 1: 0, 10: 1, 100: 2, etc...
export const fromTick = tick => Math.round(Math.log(tick) / Math.log(10))

// ex: 10.014, -2, Math.ceil
export const fixNumber = (value, precision = -2, approx = Math.round) => {
  // ex: 10.014 => 1001.4
  const shifted = value / (10 ** precision)
  // ex: 1002
  const trunc = approx(shifted)
  // ex: 10.02
  const shiftBack = trunc * (10 ** precision)

  return shiftBack.toFixed(Math.max(0, -precision))
}

export const fixNumberN = (...params) => Number(fixNumber(...params))

export const fromBig = ({ value, type = 'quote', symbol = 'BTCUSDT', approx }) => {
  if (!value)
    return 0

  if (type === 'int')
    return Number(value)

  if (!precisions[symbol][type]) {
    if (!aliases.types[type])
      throw new Error(`No precision for ${type} ${symbol}`)

    type = aliases.types[type]
  }

  const numPrecision = precisions[symbol][type]
  const bigPrecision = 10 ** (BIG_PRECISION)

  const n = Number(value) / Number(bigPrecision)
  const v = fixNumberN(n, numPrecision, approx)

  // console.log({ big, type, symbol, numPrecision, n, v })

  return v
}
export const toBig = ({ value, type, symbol = 'BTCUSDT', approx }) => {
  if (typeof value === 'bigint')
    return value

  if (type && !precisions[symbol][type]) {
    if (!aliases.types[type])
      throw new Error(`No precision for ${type} ${symbol}`)

    type = aliases.types[type]
  }

  const numPrecision = type
    ? precisions[symbol][type]
    : 0

  const f = fixNumberN(value, numPrecision, approx)
  const v = Math.round(f * 10 ** BIG_PRECISION)
  const n = BigInt(v)

  // console.log({ value, type, symbol, numPrecision, f, v, n })

  return n
}

export const formatBigEntries = (input, symbol) => {
  const precision = precisions[symbol]

  const formatted = Object.keys(input).reduce((formatted, k) => {
    const value = input[k]

    if (typeof value !== 'bigint')
      return { ...formatted, [k]: value }

    const type = precision[k] ? k : (aliases[k] || 'int')

    formatted[k] = fromBig({ type, value, symbol })

    return formatted
  }, {})

  // console.log({ input, formatted, symbol })

  return formatted
}

export default {
  values: precisions,
  fromTick,
  fixNumber,
  fixPrice,
  formatBigEntries,
  fromBig,
  toBig,
}
