import { fixPrice } from '@remora/utils/precision.mjs'

import { platforms } from '@remora/config'
import { sign } from './sign.mjs'

import updates from './updates.mjs'

import WebSocket from 'ws'

const pf = 'krk'
const {
  name: platform,
  apiKey,
  apiSecret,
  wsUrl,
} = platforms[pf]

const defaults = {
  options: {
    url: wsUrl,
  },
  books: {},
  keepAlive: {
    sent: 0,
    recv: 0,
    trip: 0,
    ival: 10 * 1000,
    tout: 5 * 1000,
    timr: null,
    ping: null,
    seen: null,
  },
}

const readyStates = [
  'CONNECTING',
  'OPEN',
  'CLOSING',
  'CLOSED',
]

const sockets = {}
const wsList = []

// TODO : Dedup
const pfSymbols = {
  PF_XBTUSD: 'BTCUSDT',
  PF_ETHUSD: 'ETHUSDT',
}
const toPfSymbol = symbol => Object.entries(pfSymbols).find(([_, pfSymbol]) => pfSymbol === symbol)[0]

const healthcheck = sock => {
  // console.log(`HEALTHCHECK ${pf}`)

  const ts = Date.now()
  const delta = ts - sock.keepAlive.seen
  if (delta < 1500) { // Pass condition
    // console.log(`HEALTHCHECK ${pf} PASS`)

    // CHAOS TEST
    // const chaos = Number(Math.random().toFixed(3))
    // const chaosTrigger = 1 / 3
    // if (chaos > chaosTrigger) {
    //   console.error(`HEALTHCHECK ${platform} FAIL : CHAOS TEST`)
    //   sock.reconnect()
    // } else
    //   console.log(`HEALTHCHECK ${platform} PASS : CHAOS TEST ${chaos} < ${Number((chaosTrigger).toFixed(3))}}`)

    return
  }

  console.warn(sock.keepAlive, `HEALTHCHECK ${pf} LATE ${delta}`)
  // console.warn(`seen: ${new Date(sock.keepAlive.seen).toISOString()}, now: ${new Date(ts).toISOString()}, delta: ${Math.round(delta / 100) / 10}s`)

  if (ts - sock.keepAlive.seen > sock.keepAlive.tout) {
    console.error(`HEALTHCHECK ${platform} FAIL`)
    sock.reconnect()
  }
}
const pingSock = sock => {
  const ts = Date.now()

  if (sock.ws.readyState !== WebSocket.OPEN) {
    const { readyState: rs } = sock.ws
    const readyState = readyStates[rs]
    console.warn(`${pf} PING ${rs}:${readyState}`)
    sock.healthcheck()
  } else { // console.log(`PING ${platform}`)
    sock.ws.ping()
    sock.keepAlive.sent = ts
  }

  sock.keepAlive.timr = setTimeout(_ => sock.healthcheck(), sock.keepAlive.tout)
}
const sockPong = sock => {
  const recv = Date.now()
  const sent = sock.keepAlive.sent
  const trip = recv - sent
  // console.log(`PONG ${platform} ${trip}`)

  // sock.keepAlive.timr = clearTimeout(sock.keepAlive.timr)

  sock.keepAlive = {
    ...sock.keepAlive,
    sent,
    recv,
    trip,
  }
}

const bookTimes = {}

const onClose = (sock, { code, why }) => {
  if (code === 1006)
    console.log(`${platform} Abnormal termination (r:"${why}" u:"${sock.ws?.url || sock.options.url}")`)
  else if (code === 1000)
    console.log(`${platform} Socket closed (r:"${why}" u:"${sock.ws?.url || sock.options.url}")`)
  else {
    console.warn(
      why.toString('utf8'),
      { readyState: sock.ws?.readyState },
    `${pf} TERMINATED ${code} ${why} ${sock.ws?.url || sock.options.url}`)
  }
}
const onError = (sock, err) => {
  console.error(err, `ERROR ${pf}}`)
  connect(sock.options)
}
const onMessage = (sock, msg) => {
  // const wsIdx = wsList.findIndex(_ws => _ws === sock.ws)
  const ts = msg.timestamp
  if (!ts && msg.event !== 'subscribed')
    console.warn(msg, `${pf} NO TIMESTAMP`)

  if (msg.event) {
    if (msg.event === 'challenge') {
      const { message: challenge } = msg
      const signed = handleChallenge(sock, challenge)

      sock.send({
        event: 'subscribe',
        feed: 'balances',
        ...signed,
      })

      sock.send({
        event: 'subscribe',
        feed: 'open_positions',
        ...signed,
      })
    }
  }

  if (msg.feed && msg.feed.startsWith('book')) {
    if (ts > sock.keepAlive.seen)
      sock.keepAlive.seen = ts

    const symbol = pfSymbols[msg.product_id]

    if (ts < bookTimes[symbol])
      console.warn(`${pf} STALE ${bookTimes[symbol] - ts} FEED ${msg.feed}.${symbol}`)
    else bookTimes[symbol] = ts

    const { seq } = msg
    const [_, type] = msg.feed.split('_')
    const partial = type === 'delta' || type !== 'snapshot'

    if (sock.books[symbol] && sock.books[symbol].handler) {
      const { timestamp } = msg
      if (type === 'snapshot') {
        sock.keepAlive.last = {
          ...sock.keepAlive.last || {},
          [symbol]: seq,
        }
        const extractedData = {
          asks: msg.asks.map(({ price, qty }) => [price, qty]),
          bids: msg.bids.map(({ price, qty }) => [price, qty]),
          timestamp,
          partial,
          platform,
          symbol,
        }
        sock.books[symbol].handler(extractedData)
      } else if (partial) {
        const { [symbol]: last } = sock.keepAlive.last || {}
        if (last && seq > last + 1) {
          console.error(`${pf} SKIP ${last - seq} ${last}-${seq}`)
          sock.reconnect()
        }
        if (!last || seq !== last + 1)
          return console.log(`${platform}: Ignoring seq ${seq} last: ${last}`)
        else {
          sock.keepAlive.last = {
            ...sock.keepAlive.last,
            [symbol]: seq,
          }
        }

        const extractedData = {
          asks: msg.side === 'sell' ? [[fixPrice(msg.price, symbol), msg.qty.toString()]] : [],
          bids: msg.side === 'buy' ? [[fixPrice(msg.price, symbol), msg.qty.toString()]] : [],
          timestamp,
          partial,
          platform,
          symbol,
        }
        sock.books[symbol].handler(extractedData)
      }
    }
  } else {
    const translate = updates.match(msg)

    if (translate) {
      const translated = translate(msg)
      const { type, data } = translated

      sock.keepAlive.seen = data.timestamp

      if (sock[type].listeners)
        sock[type].listeners.forEach(listener => listener(data))
      else
        console.warn(`${platform} - No listener for message ${type}`)
    } else
      console.warn(`Unhandled message on ${platform}`)
  }
}

const bookSub = (sock, symbol, cb) => {
  console.log(`SUB ${platform} ${symbol}`)
  if (!sock.books[symbol])
    sock.books[symbol] = { handler: cb }
  else if (cb)
    sock.books[symbol].handler = cb

  const payload = {
    event: 'subscribe',
    feed: 'book',
    product_ids: [toPfSymbol(symbol)],
  }

  sock.send(payload)
}
const bookUnsub = (sock, symbol) => {
  const payload = {
    event: 'unsubscribe',
    feed: 'book',
    product_ids: [toPfSymbol(symbol)],
  }
  sock.send(payload)
  delete sock.books[symbol]
}

const auth = async (sock, credentials = { apiKey, apiSecret }) => {
  // console.log({ credentials })
  if (!credentials.apiKey || !credentials.apiSecret)
    throw new Error(`Missing credentials for ${platform}`)

  const payload = {
    event: 'challenge',
    api_key: credentials.apiKey,
  }

  sock.options.credentials = { ...credentials }
  sock.send(payload)

  return sock
}
const accountSub = (sock, ev, cb) => {
  const { apiKey } = sock.options.credentials
  if (!apiKey)
    throw new Error(`Missing credentials for ${platform}`)

  if (!sock.account.listeners)
    sock.account.listeners = []

  if (ev === 'update') {
    sock.account.listeners.push(cb)
    console.info(`${pf}::account:subscribed`, sock.account.listeners.length)
  } else
    console.warn(`BADSUB ${pf} ${ev}`)
}

const disconnect = (sock, cb) => {
  const noop = _ => {}
  if (!cb)
    cb = noop

  console.log(`DISCONNECT ${platform}`)
  if (sock.ws) {
    if (sock.ws.readyState < 1) {
      sock.ws.removeAllListeners('error')
      sock.ws.on('error', noop)
      sock.ws.removeAllListeners('close')
      sock.ws.on('close', cb)
    }

    sock.ws.close()
    delete (sock.ws)
  }
  clearTimeout(sock.keepAlive.timr)
  sock.keepAlive.timr = null

  clearInterval(sock.keepAlive.ping)
  sock.keepAlive.ping = null

  sock.keepAlive.nonce = 0
}

let connectFailCount = 0
export const connect = options => {
  options = { ...defaults.options, ...options }
  let { url } = options

  const socket = sockets[url] || { ...defaults }
  if (sockets[url])
    socket.disconnect()

  socket.options = { ...socket.options, ...options }
  url = options.url

  console.log(`CONNECT ${pf}`)

  if (connectFailCount)
    console.log(`CONNECT ${pf} ${connectFailCount}`)

  const onConnectError = err => {
    connectFailCount++

    console.error(`CONNECT ${pf} FAIL`, err)
    wsList.splice(wsList.findIndex(_ws => _ws === ws), 1)

    setTimeout(_ => {
      console.log(`CONNECT ${pf} RETRY`)
      connect(socket.options)
    }, connectFailCount * 250)
  }
  const ws = socket.ws = new WebSocket(url)
  wsList.push(ws)
  ws.on('error', onConnectError)

  ws.on('message', data => {
    try {
      const message = JSON.parse(data)
      socket.onMessage(message)
    } catch (error) {
      console.error(error)
      socket.reconnect()
    }
  })
  ws.on('pong', _ => socket.pong())
  ws.on('open', _ => {
    connectFailCount = 0

    ws.off('error', onConnectError)
    ws.on('error', err => { socket.onError(err) })
    // ws.on('close', r => onClose(socket, r))
    ws.on('close', (code, why) => {
      onClose(socket, { code, why })
      wsList.splice(wsList.findIndex(_ws => _ws === ws), 1)
    })

    // if (socket.ws) socket.ws.terminate()

    // socket.ws = ws

    socket.keepAlive.ping = setInterval(_ => socket.ping(), socket.keepAlive.ival)

    console.log(`CONNECTED ${pf}`)
  })

  socket.send = msg => {
    if (typeof msg !== 'string') {
      // if (socket.options.credentials?.original_challenge)
      //   msg = { ...msg, ...socket.options.credentials }

      msg = JSON.stringify(msg)
    }

    if (socket.ws.readyState < WebSocket.OPEN) {
      socket.ws.on('open', _ => {
        try {
          console.log(`${JSON.stringify(JSON.parse(msg), null, 2)} ${pf} SEND`)
          socket.ws.send(msg)
        } catch (err) {
          console.error(`error ${pf} send`, err)
          console.log(wsList.map(({ url, readyState }) => ({ url, readyState })))
          throw err
        }
        // socket.ws.send(msg)
      })
    } else if (socket.ws.readyState === WebSocket.OPEN)
      socket.ws.send(msg)
    else
      throw new Error(`${platform} socket not ready : ${socket.ws.readyState}`)
  }

  socket.healthcheck = _ => healthcheck(socket)
  socket.pong = msg => sockPong(socket, msg)
  socket.ping = _ => {
    try {
      pingSock(socket)
    } catch (err) {
      console.error(`error ${pf} ping`, err)
    }
  }
  socket.onError = err => onError(socket, err)
  socket.onMessage = msg => onMessage(socket, msg)
  socket.reconnect = options => {
    options = { ...socket.options, ...options }
    console.log(`CONNECT ${pf} RE`)
    return connect(options)
  }
  socket.disconnect = _ => disconnect(socket)

  socket.book = {
    on: (symbol, cb) => bookSub(socket, symbol, cb),
    off: symbol => bookUnsub(socket, symbol),
  }
  socket.account = {
    ...socket.account,
    on: (ev, cb) => { console.log(`SUB ${pf} ACCOUNT`); accountSub(socket, ev, cb) },
    // off: (ev, cb) => accountUnsub(socket, ev, cb),
  }

  Object.entries(socket.books).forEach(([symbol, sub]) => {
    socket.book.on(symbol, sub.handler)
  })

  socket.auth = credentials => auth(socket, credentials)
  if (socket.options.credentials)
    socket.auth()

  return (sockets[url] = socket)
}

export default {
  connect,
  // list: _ => sockets,
  // auth,
}

const handleChallenge = (sock, challenge) => {
  const { apiKey, apiSecret } = sock.options.credentials

  const signed = sign(challenge, apiSecret)

  return {
    api_key: apiKey,
    original_challenge: challenge,
    signed_challenge: signed,
  }
}

// setInterval(_ => {
//   const open = wsList.filter(ws => ws.readyState === WebSocket.OPEN)

//   if (open.length > 1)
//     console.warn(`${pf} MULTIPLE OPEN SOCKETS`)

//   if (wsList.length > 1)
//     console.warn(`${pf} MULTIPLE SOCKETS`)

//   // console.log(sockets)
// }, 10 * 1000)
