import { dump } from '@remora/utils'

const platform = 'Kraken'

const feeds = {
  balances: 'balances',
  open_positions: 'positions',
}

const formatBalances = data => {
  const { timestamp } = data

  const currencies = data.flex_futures?.currencies || {}
  if (!data.flex_futures)
    dump(data, 'no currencies in flex_futures !?')
  const balances = Object.keys(currencies)
    .sort((a, b) => currencies[b].quantity - currencies[a].quantity)
    .reduce((balances, currency) => {
      const balance = currencies[currency]

      if (balance.quantity) {
        // How Kraken computes margin :
        // Haircut = how much of the balance is taken into account.
        // Shitcoins have high haircut, stablecoins low.
        // Conversion rate :
        // Not shown in the balance, but can be inferred from quantity and value.
        // For example, if we're using 1M $SHIT as collat, with following values :
        // haircut: 0.5 (50%), conversion: 0.001 (1thou of a USD)
        // We'll have 500K into collat (from haircut) valued at 0.5USD
        const haircutRate = 1 - balance.haircut
        const conversionRate = balance.value / balance.quantity
        const marginRate = haircutRate * conversionRate

        // Convert available balance of asset into available collat from said asset
        // This is needed because kraken's data shows available before collateral calculation,
        // and collateral_value before "used" calculation.
        const available = balance.available * marginRate
        const total = balance.quantity

        balances[currency] = {
          total,
          available,
          lastUpdated: balance.lastUpdated || timestamp,
          extraInfo: { ...balance },
        }
      }

      return balances
    }, {})

  const formatted = {
    timestamp,
    balances,
  }
  // console.log(formatted)

  return ({
    type: 'account',
    data: formatted,
  })
}

// TODO : Dedup ?
const pfSymbols = {
  PF_XBTUSD: 'BTCUSDT',
  PF_ETHUSD: 'ETHUSDT',
}

const formatPositions = data => {
  const { timestamp } = data

  // console.log(data)

  const positions = data.positions.reduce((positions, position) => {
    const symbol = pfSymbols[position.instrument]
    const {
      balance: quantity,
      entry_price: entry,
      max_fixed_leverage: leverage,
      initial_margin: margin,
      ...extraInfo
    } = position

    return ({
      ...positions,
      [symbol]: {
        quantity,
        entry,
        margin,
        leverage,
        extraInfo,
        lastUpdated: position.lastUpdated ||
          position.timestamp ||
          timestamp,
      },
    })
  }, {})

  const formatted = {
    timestamp,
    positions,
  }

  return {
    type: 'account',
    data: formatted,
  }
}

export const match = msg => {
  // console.log(msg, `in:match ${platform}`)
  const handledBy = feeds[msg.feed]
  if (!handledBy)
    return

  if (msg.event === 'subscribed') {
    console.info(`${platform} SUBSCRIBE ${msg.feed}`)
    return
  }

  if (!handlers[handledBy]) {
    console.warn(`${platform}::warn No handler for ${msg.feed}`)

    return
  }

  return handlers[handledBy]
}

const handlers = {
  balances: formatBalances,
  positions: formatPositions,
}

export default {
  match,
}
