import { resolve } from 'node:path/posix'

import { platforms } from '@remora/config'

import { authenticate } from './sign.mjs'

import ws from './websocket.mjs'
import { toQueryString } from '@remora/utils/queryString.mjs'
import { dump } from '@remora/utils'

// Kraken API
export const pf = 'krk'

// TODO : Dedup
const pfSymbols = {
  PF_XBTUSD: 'BTCUSDT',
  PF_ETHUSD: 'ETHUSDT',
}
const toPfSymbol = symbol => Object.entries(pfSymbols).find(([_, pfSymbol]) => pfSymbol === symbol)[0]

const config = { ...platforms[pf] }
const {
  apiKey,
  apiSecret,
  baseUrl,
} = config

console.log(config.name, baseUrl)

const defaults = {
  method: 'POST',
  headers: {
    contentType: 'application/json',
    'X-MBX-APIKEY': apiKey,
  },
}

export const get = (path, signed) =>
  request(path, { method: 'GET' }, signed)

export const post = (path, options, signed = true) =>
  request(path, {
    ...defaults,
    ...options,
  }, signed)

export const put = (path, options, signed = true) =>
  request(path, {
    method: 'PUT',
    ...options,
  }, signed)

export const del = (path, options, signed = true) =>
  request(path, {
    method: 'DELETE',
    ...options,
  }, signed)

export const request = (path, options = defaults, signed = true) => {
  // const timestamp = Date.now()
  options = Object.assign({}, defaults, options)

  if (typeof options === 'boolean') {
    signed = options
    options = defaults
  }

  const [urlPath, queryString] = path.split('?')
  const url = new URL(baseUrl)

  url.pathname = resolve(url.pathname, urlPath)

  if (queryString) {
    const queryParams = Object.fromEntries(new URLSearchParams(queryString))
    options.params = { ...options.params, ...queryParams }
  }

  const headers = {
    'Content-Type': 'application/json',
  }

  if (options.headers)
    Object.assign(headers, options.headers)

  // if (options.body)
  //   options.body = toQueryString(options.body)

  // if (!options.body?.timestamp && !options.params?.timestamp)
  //     options.params = { ...options.params, timestamp }

  if (signed)
    options = authenticate({ apiKey, apiSecret, ...options }, url.pathname)

  if (options.params)
    url.search = `?${new URLSearchParams(options.params).toString()}`

  console.log(JSON.stringify({ url, options }, null, 2))

  return fetch(url, options)
    .then(response => {
      const contentType = response.headers.get('content-type')
      if (!contentType || !contentType.includes('application/json')) {
        response.text().then(body => {
          console.error(body, contentType)
          throw new Error('Response was not in JSON format')
        })
      }
      return response.json()
    })
    .catch(error => console.error(error))
}

const init = (cfg = {}) => Object.assign(config, cfg)

const getAssets = () => {
  return request('api/v1/private/account/assets')
}

const order = ({ symbol, side, quantity, price }) => {
  const orderParams = {
    // orderType: 'mkt',
    orderType: 'lmt',
    limitPrice: price,
    side,
    symbol: toPfSymbol(symbol),
    size: Number(quantity),
    // reduceOnly: false,
  }

  return post('sendorder', {
    headers: {
      'Content-Type': 'application/x-www-form-urlencoded',
    },
    body: toQueryString(orderParams),
  }).then(response => {
    console.log(response.result)
    if (response.result !== 'success') {
      const e = new Error('Fulfilment error.')
      e.detail = response
      throw e
    }

    return response
  }).then(({ sendStatus, serverTime }) => {
    if (!sendStatus || sendStatus.status !== 'placed') {
      const e = new Error('Execution error.')
      e.detail = { sendStatus, serverTime }
      dump(e.detail)
      throw e
    }

    return sendStatus
  })
}

export default {
  platform: pf,
  name: 'Kraken',
  config: init,
  init,
  getAssets,
  request,
  get,
  post,
  put,
  delete: del,
  ws,
  order,
}
