import WebSocket from 'ws'
import {
  fixNumber,
  fixPrice,
  fromTick,
} from '@remora/utils/precision.mjs'

const platform = 'KuCoin'

const defaults = {
  url: 'https://api-futures.kucoin.com',
  options: {
    ws: {},
  },
  keepAlive: {
    sent: 0,
    recv: 0,
    trip: 0,
    ival: 5 * 1000,
    tout: 3.5 * 1000,
    timr: null,
    ping: null,
    seen: null,
  },
}

const sockets = {}

const pfSymbols = {
  XBTUSDTM: 'BTCUSDT',
  ETHUSDTM: 'ETHUSDT',
}
const toPfSymbol = symbol => Object.entries(pfSymbols).find(([_, pfSymbol]) => pfSymbol === symbol)[0]

const lotSizes = {
  BTCUSDT: 0.001,
  ETHUSDT: 0.01,
}
const volPrecisions = {
  BTCUSDT: fromTick(lotSizes.BTCUSDT),
  ETHUSDT: fromTick(lotSizes.ETHUSDT),
}
const toQty = (lots, symbol) => fixNumber(lots * lotSizes[symbol], volPrecisions[symbol])

const healthcheck = sock => {
  sock.keepAlive.timr = clearTimeout(sock.keepAlive.timr)
  const ts = Date.now()
  const delta = ts - sock.keepAlive.seen
  if (delta < 1000)
    return

  console.warn(sock.keepAlive, `HEALTHCHECK: ${platform} ?!`)
  console.warn(`seen: ${sock.keepAlive.seen}, now: ${ts}, delta: ${Math.round(delta / 100) / 10}`)

  if (ts - sock.keepAlive.seen > sock.keepAlive.tout) {
    console.error(`${platform} socket dead !?`)
    sock.reconnect()
  }
}
const pingSock = sock => {
  const ts = Date.now()
  const ping = { id: ts, type: 'ping' }
  // console.log(`PING ${platform}`)
  sock.send(JSON.stringify(ping))
  // sock.ws?.ping()
  sock.keepAlive.sent = ts

  sock.keepAlive.timr = setTimeout(_ => sock.healthcheck(), sock.keepAlive.tout)
}
const sockPong = (sock, pong = {}) => {
  const recv = Date.now()
  const sent = +pong.id
  const trip = recv - sent
  // console.log(`PONG ${platform} TRIP: ${trip}`)

  // sock.keepAlive.timr = clearTimeout(sock.keepAlive.timr)

  sock.keepAlive = {
    ...sock.keepAlive,
    sent,
    recv,
    trip,
  }

  // sock.keepAlive.ping = setTimeout(_ => sock.ping(), sock.keepAlive.ival)
}

const onClose = (sock, why) => {
  console.warn(`SOCK CLOSED: ${platform}`, why, { connecting: sock.connecting })
  if (!sock.connecting && why !== 1006)
    sock.reconnect()
}
const onError = (sock, err) => {
  console.error(err, `<- WS error (${platform})`)
  setImmediate(sock.reconnect)
}
const onMessage = (sock, msg) => {
  const T = Date.now()
  if (msg.type === 'pong')
    sock.pong(msg)

  else if (msg.topic) {
    if (T > sock.keepAlive.seen)
      sock.keepAlive.seen = T

    if (msg.topic.startsWith('/contractMarket/level2Depth50:')) {
      const [_, pfSymbol] = msg.topic.split(':')
      const symbol = pfSymbols[pfSymbol]
      const partial = false

      if (sock.books[symbol] && sock.books[symbol].handler) {
        sock.books[symbol].handler({
          asks: msg.data.asks.map(([price, lots]) => [fixPrice(price, symbol), toQty(lots, symbol)]),
          bids: msg.data.bids.map(([price, lots]) => [fixPrice(price, symbol), toQty(lots, symbol)]),
          timestamp: msg.data.timestamp,
          partial,
          platform,
          symbol,
        })
      }
    }
  } else
    console.log(msg, `Unhandled message on ${platform}`)
}

const bookSub = (sock, symbol, cb) => {
  console.log(`SUB ${platform} ${symbol}`)
  if (!sock.books[symbol])
    sock.books[symbol] = { handler: cb }
  else if (cb)
    sock.books[symbol].handler = cb

  const payload = {
    id: sock.id,
    type: 'subscribe',
    topic: `/contractMarket/level2Depth50:${toPfSymbol(symbol)}`,
  }
  // sock.connected.then(_ => sock.send(JSON.stringify(payload))
  sock.send(JSON.stringify(payload))
}
const bookUnsub = (sock, symbol) => {
  const payload = {
    id: sock.id,
    type: 'unsubscribe',
    topic: `/contractMarket/level2depth50:${toPfSymbol(symbol)}`,
  }
  sock.send(JSON.stringify(payload))
  delete sock.books[symbol]
}

let connectFailCount = 0
export const connect = (url, options = defaults.options) => {
  console.log(`CONNECT ${platform}`)
  url ||= defaults.options.url || defaults.url
  const path = options?.path ||
        `/api/v1/bullet-${options?.private ? 'private' : 'public'}`

  const socket = sockets[url] || {
    options: {
      ...defaults.options,
      ...options,
      url,
      path,
    },
    isReady: false,
    book: null,
    books: {},
    keepAlive: { ...defaults.keepAlive },
  }

  if (socket.connecting)
    return socket

  socket.connecting = true

  if (socket.ws) {
    // socket.ws.close()
    // delete (socket.ws)
    clearInterval(socket.keepAlive.ping)
    clearTimeout(socket.keepAlive.timr)
    socket.keepAlive = {
      ...defaults.keepAlive,
      seen: Date.now(),
    }
  }

  socket.connected = new Promise((resolve, reject) => {
    if (connectFailCount)
      console.log(`${platform} CONNECT ATTEMPT ${connectFailCount}`)

    fetch(`${url}${path}`, { method: 'POST' }).then(response => response.json())
      .then(({
        data: {
          token,
          instanceServers: [server],
        },
      }) => {
        const WSS_URL = `${server.endpoint}?token=${token}&[connectId=${token}]`

        socket.keepAlive.tout = server.pingTimeout
        const ws = new WebSocket(WSS_URL)

        ws.on('error', reject)
        const handleFirstMessage = msg => {
          const data = JSON.parse(msg)
          if (data.type === 'welcome') {
            socket.id = data.id
            ws.on('message', data => socket.onMessage(JSON.parse(data)))

            connectFailCount = 0

            if (socket.ws) socket.ws.terminate()

            socket.ws = ws

            socket.keepAlive.ping = setInterval(_ => socket.ping(), socket.keepAlive.ival)

            console.log(`WS connection established : ${url}`)
            resolve(ws)
          }
        }

        ws.on('open', _ => {
          ws.on('error', err => { socket.onError(err) })
          ws.on('message', handleFirstMessage)
          ws.on('close', r => onClose(socket, r))
        })
      })
  }).catch(error => {
    console.error(`Connecting on ${platform} failed.`, error)
    connectFailCount++
    setTimeout(_ => { connect(_, socket.options) }, connectFailCount * 500)
  }).finally(_ => {
    socket.connecting = false
  })

  socket.send = msg => {
    socket.connected.then(_ => {
      try { socket.ws.send(msg) } catch (e) { console.error(e, msg) }
    })
  }

  socket.healthcheck = _ => healthcheck(socket)
  socket.pong = msg => sockPong(socket, msg)
  socket.ping = _ => pingSock(socket)
  socket.onError = err => onError(socket, err)
  socket.onMessage = msg => onMessage(socket, msg)
  socket.reconnect = _ => connect(null, socket.options)

  socket.book = {
    on: (symbol, cb) => bookSub(socket, symbol, cb),
    off: symbol => bookUnsub(socket, symbol),
  }

  Object.entries(socket.books)
    .forEach(([symbol, sub]) => socket.book.on(symbol, sub.handler))

  return (sockets[url] = socket)
}

export default {
  connect,
  list: _ => sockets,
}
