import { generateHeaders } from './sign.mjs'

import ws from './websocket.mjs'

let apiKey = process.env.EXCHANGE_API_KEY
let apiSecret = process.env.EXCHANGE_API_SECRET
let baseUrl = 'https://contract.mexc.com'

const request = (path, options = {}, signed = true) => {
  if (typeof options === 'boolean') {
    signed = options
    options = {}
  }

  const reqTime = Date.now()

  const [urlPath, queryString] = path.split('?')
  const url = new URL(urlPath, baseUrl)

  if (queryString) {
    const queryParams = Object.fromEntries(new URLSearchParams(queryString))
    options.params = { ...options.params, ...queryParams }
  }

  const headers = {
    'Request-Time': reqTime,
    'Content-Type': 'application/json',
  }

  if (options.headers)
    Object.assign(headers, options.headers)

  if (signed) {
    const method = options.method || 'GET'
    const queryParams = options.params
    const body = options.body
    const authHeaders = generateHeaders({
      apiKey,
      apiSecret,
      reqTime,
      path: urlPath,
      method,
      queryParams,
      body,
    })
    Object.assign(headers, authHeaders)
  }

  const fetchOptions = {
    method: options.method || 'GET',
    headers,
    body: options.body ? JSON.stringify(options.body) : undefined,
  }

  if (options.params)
    url.search = `?${new URLSearchParams(options.params).toString()}`

  return fetch(url, fetchOptions)
    .then(response => {
      const contentType = response.headers.get('content-type')
      if (!contentType || !contentType.includes('application/json')) {
        response.text().then(body => {
          console.error(body, contentType)
          throw new Error('Response was not in JSON format')
        })
      }
      return response.json()
    })
    .catch(error => console.error(error))
}

const init = (cfg = {}) => {
  apiKey ||= cfg.apiKey
  apiSecret ||= cfg.apiSecret
  baseUrl ||= cfg.baseUrl
}

const getAssets = () => {
  return request('api/v1/private/account/assets')
}

export default {
  config: init,
  init,
  getAssets,
  request,
  ws,
}
