import WebSocket from 'ws'

import { platforms } from '@remora/config'
import { sign } from './sign.mjs'
import updates from './updates.mjs'
import { getAssets, getPositions } from './client.mjs'

const noop = _ => {}

const pf = 'byb'
const {
  name: platform,
  apiKey,
  apiSecret,
  wsUrl,
} = platforms[pf]

const defaults = {
  options: {
    url: wsUrl,
  },
  book: null,
  books: {},
  account: {},
  keepAlive: {
    sent: 0,
    recv: 0,
    trip: 0,
    ival: 5 * 1000,
    tout: 3.5 * 1000,
    timr: null,
    ping: null,
    seen: null,
    nonce: 0,
  },
}

const sockets = {}
const wsList = []

// Kill in 10 secs to test reconnection
// setTimeout(_ => { Object.values(sockets).at(-1).ws.close() }, 10000)

const healthcheck = sock => {
  const reportHealthcheck = msg => console.log(`HEALTHCHECK ${msg} ${platform} ${sock.options.url} ${sock.ws.url}`)
  const ts = Date.now()
  const { recv, seen, sent, tout: timeout } = sock.keepAlive

  if (!seen || !recv) {
    reportHealthcheck('SOCKET NOT READY')
    if (sock.ws.readyState > 1) {
      console.warn(`${platform} socket not ready for healthcheck`, sock.ws.readyState)
      sock.reconnect()
    }
    return
  }

  // const delta = ts - sock.keepAlive.seen
  const delta = Math.min(ts - seen, ts - recv)

  if (delta > timeout) {
    reportHealthcheck('SOCKET TIMEOUT')
    console.error({
      sou: sock.options.url,
      wsu: sock.ws.url,
      ts,
      recv,
      sent,
      seen,
      delta,
      timeout,
    })
    sock.reconnect()
  }
}
const pingSock = sock => {
  const ts = Date.now()
  const payload = { op: 'ping' }

  sock.send(payload, msg => sock.pong(msg, ts))
  sock.keepAlive.sent = ts

  sock.keepAlive.timr = setTimeout(_ => sock.healthcheck(), sock.keepAlive.tout)
}
const sockPong = (sock, _pong, sent = 0) => {
  const recv = Date.now()
  const trip = recv - sent

  sock.keepAlive = {
    ...sock.keepAlive,
    sent,
    recv,
    trip,
  }
}

const onClose = (sock, code, why) => {
  if (code === 1006)
    console.log(`${platform} Abnormal termination (r:"${why}" u:"${sock.ws?.url || sock.options.url}")`)
  else if (code === 1000)
    console.log(`${platform} Socket closed (r:"${why}" u:"${sock.ws?.url || sock.options.url}")`)
  else {
    console.warn(
      why.toString('utf8'),
      { readyState: sock.ws?.readyState },
      `Unexpected sock close : ${platform}`)
  }
}
const onError = (sock, err) => {
  console.error(err, `<- WS error (${platform})`)
  sock.reconnect()
}
const onMessage = (sock, msg) => {
  if (msg.req_id) {
    const { req_id: id } = msg

    if (sock.responses[id]) {
      const { [id]: handler, ...handlers } = sock.responses
      sock.responses = handlers
      handler(msg)
    }
  } else if (msg.topic && msg.topic.startsWith('orderbook.')) {
    const { ts: timestamp } = msg
    if (timestamp > sock.keepAlive.seen)
      sock.keepAlive.seen = timestamp

    const [_1, _2, symbol] = msg.topic.split('.')
    const { u: updateId } = msg.data

    if (msg.type === 'snapshot') {
      sock.keepAlive.last = {
        ...sock.keepAlive.last || {},
        [symbol]: updateId,
      }
    } else if (msg.type === 'delta') {
      if (!sock.keepAlive.last)
        sock.keepAlive.last = {}

      const { [symbol]: last } = sock.keepAlive.last

      if (last && updateId > last + 1)
        sock.reconnect()
      if (updateId !== last + 1)
        return console.error(`${platform} ${symbol} update out of sync. Last: ${last} -> ${updateId}`)
      if (updateId === last + 1)
        sock.keepAlive.last[symbol] = updateId
    } else {
      console.log(msg, `Unhandled message on ${platform}`)
      return
    }
    const { a: asks, b: bids } = msg.data

    if (sock.books[symbol] && sock.books[symbol].handler) {
      const partial = msg.type !== 'snapshot'

      sock.books[symbol].handler({
        asks,
        bids,
        timestamp,
        partial,
        platform,
        symbol,
      })
    }
  } else {
    const translate = updates.match(msg)

    if (translate) {
      const translated = translate(msg)
      const { type, data } = translated

      sock.keepAlive.seen = data.timestamp

      if (sock[type] && sock[type].listeners)
        sock[type].listeners.forEach(listener => listener(data))
      else
        console.warn(`${platform} - No listener for message ${type}`)
    } else
      console.warn(msg, `Unhandled message on ${platform}`)
  }
}

const bookSub = (sock, symbol, cb) => {
  console.log(`SUB ${platform} ${symbol}`)
  if (!sock.books[symbol]) {
    sock.books = {
      ...sock.books,
      [symbol]: { handler: cb },
    }
  } else if (cb)
    sock.books[symbol].handler = cb

  if (sock.ws.readyState === 1) {
    const payload = {
      op: 'subscribe',
      args: [`orderbook.50.${symbol}`],
    }
    sock.send(payload, msg => {
      console.log(msg, sock.ws.url)
    })
  }
}
const bookUnsub = (sock, symbol) => {
  const payload = {
    op: 'unsubscribe',
    args: [`orderbook.50.${symbol}`],
  }
  sock.send(payload)
  delete sock.books[symbol]
}

const auth = async (sock, credentials = { apiKey, apiSecret }) => {
  if (!credentials.apiKey || !credentials.apiSecret)
    throw new Error(`Missing credentials for ${platform}`)

  if (sock.ws.url.includes('public')) {
    const altSock = connect({ credentials })
    sock.account = altSock.account

    return sock
  }

  const timeout = Date.now() + (2.5 * 1000)
  const challenge = `GET/realtime${timeout}`
  const sig = sign(challenge, credentials.apiSecret)

  const payload = {
    op: 'auth',
    args: [
      apiKey,
      timeout,
      sig,
    ],
  }

  const onResponse = msg => {
    console.log(msg, 'AUTH RESPONSE')
    if (msg.success === true) {
      sock.send({
        op: 'subscribe',
        args: [
          'position',
          'wallet',
        ],
      })
    } else {
      console.error(`${platform} FAILED AUTH`)
      process.exit(-1)
    }
  }

  await Promise.all([
    getAssets().then(({ time, result }) => {
      const msg = {
        topic: 'wallet',
        creationTime: time,
        data: result.list,
      }
      console.log(JSON.stringify(result, null, 2), 'HTTP FETCH ASSETS')

      sock.onMessage(msg)
    }),
    getPositions().then(({ time, result }) => {
      console.log(JSON.stringify(result, null, 2), 'HTTP FETCH POSITIONS')
      const msg = {
        topic: 'position',
        creationTime: time,
        data: result.list,
      }

      sock.onMessage(msg)
    }),
  ]).then(_ => { sock.send(payload, onResponse) })

  return sock
}

const accountSub = (sock, ev, cb) => {
  const { apiSecret } = sock.options.credentials
  if (!apiSecret)
    throw new Error(`Missing credentials for ${platform}`)

  if (!sock.account.listeners)
    sock.account.listeners = []

  if (ev === 'update') {
    sock.account.listeners.push(cb)
    console.info(`${pf}::account:subscribed`, sock.account.listeners.length)
  } else
    console.warn(`BADSUB ${pf} ${ev}`)
}
const accountUnsub = (sock, ev, cb) => {}

const disconnect = (sock, cb) => {
  if (!cb) cb = noop

  console.log(`DISCONNECT ${platform}`)
  if (sock.ws) {
    if (sock.ws.readyState < 1) {
      sock.ws.removeAllListeners('error')
      sock.ws.on('error', noop)
      sock.ws.removeAllListeners('close')
      sock.ws.on('close', cb)
    }

    sock.ws.close()
    delete sock.ws
  }
  clearTimeout(sock.keepAlive.timr)
  sock.keepAlive.timr = null

  clearInterval(sock.keepAlive.ping)
  sock.keepAlive.ping = null

  sock.keepAlive.nonce = 0
}

let connectFailCount = 0
export const connect = options => {
  options = { ...defaults.options, ...options }
  let { url } = options

  if (options.credentials && !url.includes('/v5/private'))
    url += '/v5/private'
  else if (!options.credentials && !url.includes('/v5/public/linear'))
    url += '/v5/public/linear'

  const socket = sockets[url] || { ...defaults }

  if (sockets[url])
    socket.disconnect()

  socket.options = { ...socket.options, ...options }
  if (socket.options.url !== url) {
    console.log('URL', platform, url)
    socket.options.url = url
  }

  console.log(`CONNECT ${platform} ${options.credentials ? 'PRIVATE' : 'PUBLIC'}`)

  socket.healthcheck = _ => healthcheck(socket)
  socket.pong = msg => sockPong(socket, msg)
  socket.ping = _ => pingSock(socket)
  socket.onError = err => onError(socket, err)
  socket.onMessage = msg => onMessage(socket, msg)
  socket.reconnect = options => {
    options = { ...socket.options, ...options }
    console.log(`CONNECT ${platform} RE`)

    return connect(options)
  }
  socket.disconnect = _ => disconnect(socket)

  socket.book = {
    on: (symbol, cb) => bookSub(socket, symbol, cb),
    off: symbol => bookUnsub(socket, symbol),
  }
  socket.account = {
    on: (ev, cb) => accountSub(socket, ev, cb),
    off: (ev, cb) => accountUnsub(socket, ev, cb),
  }

  if (connectFailCount) {
    console.log(`CONNECT ${pf} ${connectFailCount}`)
    if (connectFailCount > 10) {
      console.log(`${platform} giving up : failed connecting ${connectFailCount} times.`)
      process.exit(-1)
    }
  }

  if (socket.ws)
    socket.ws.close()

  const ws = socket.ws = new WebSocket(url)
  wsList.push(ws)

  const onConnectError = err => {
    connectFailCount++

    console.error(err, `<- WS error in connect (${platform})`)
    wsList.splice(wsList.findIndex(_ws => _ws === ws), 1)

    console.error(`Connecting on ${platform} failed.`, err)
    setTimeout(_ => {
      console.log(`CONNECT ${platform} RETRY`)
      socket.reconnect()
    }, connectFailCount * 250)
  }
  ws.on('error', onConnectError)
  ws.on('message', data => socket.onMessage(JSON.parse(data)))
  ws.on('close', (code, why) => {
    onClose(socket, code, why)
    wsList.splice(wsList.findIndex(_ws => _ws === ws), 1)
  })

  ws.on('open', _ => {
    ws.off('error', onConnectError)
    ws.on('error', err => {
      socket.onError(err)
    })

    connectFailCount = 0

    if (socket.keepAlive.ping) clearInterval(socket.keepAlive.ping)

    socket.keepAlive = {
      ...socket.keepAlive,

      sent: Date.now(),
      seen: Date.now(),
    }
    socket.keepAlive.ping = setInterval(_ => socket.ping(), socket.keepAlive.ival)

    Object.entries(socket.books).forEach(([symbol, sub]) => {
      socket.book.on(symbol, sub.handler)
    })

    console.log(`CONNECTED ${platform} ${options.credentials ? 'PRIVATE' : 'PUBLIC'}`)
  })

  socket.send = (msg, onResponse) => {
    if (typeof socket.keepAlive.nonce !== 'number' || Number.isNaN(socket.keepAlive.nonce))
      console.error('NONCE NOT A NUMBER', socket.keepAlive.nonce)

    if (!socket.ws)
      return console.error('No socket in send', msg)

    if (socket.ws.readyState !== 1) {
      socket.ws.on('open', _ => {
        socket.send(msg, onResponse)
      })
      return
    }

    let id
    if (typeof msg !== 'string') {
      id = ++socket.keepAlive.nonce

      msg = JSON.stringify({
        ...msg,
        req_id: id,
      })
    }

    socket.ws.send(msg)

    if (id && onResponse) {
      if (socket.responses?.[id])
        throw new Error(`${platform} ws request nonce collision ${id}`)

      socket.responses = {
        ...socket.responses,
        [id]: onResponse,
      }

      setTimeout(_ => {
        if (!socket.responses) {
          const fault = new Error('No responses in socket')
          console.error(socket)
          throw fault
        }
        if (socket.responses[id])
          console.error(`Response timeout ${id}`)
      }, socket.keepAlive.tout)
    }
  }

  socket.auth = credentials => auth(socket, credentials)
  if (socket.options.credentials)
    socket.auth()

  return (sockets[url] = socket)
}

export default {
  connect,
  // list: _ => sockets,
}
