import { resolve } from 'node:path/posix'

import { platforms } from '@remora/config'

import { authenticate } from './sign.mjs'

import ws from './websocket.mjs'

// Bybit API
export const pf = 'byb'

const config = { ...platforms[pf] }
const {
  apiKey,
  apiSecret,
  baseUrl,
} = config

console.log(config.name, baseUrl)

const defaults = {
  method: 'POST',
  headers: {
    contentType: 'application/json',
  },
}

export const get = (path, options, signed = true) =>
  request(path, {
    ...defaults,
    ...options,
    method: 'GET',
  }, signed)

export const post = (path, options, signed = true) =>
  request(path, {
    ...defaults,
    ...options,
    method: 'POST',
  }, signed)

export const put = (path, options, signed = true) =>
  request(path, {
    method: 'PUT',
    ...options,
  }, signed)

export const del = (path, options, signed = true) =>
  request(path, {
    method: 'DELETE',
    ...options,
  }, signed)

export const request = (path, options = defaults, signed = true) => {
  // const timestamp = Date.now()
  options = Object.assign({}, defaults, options)

  if (typeof options === 'boolean') {
    signed = options
    options = defaults
  }

  const [urlPath, queryString] = path.split('?')
  const url = new URL(baseUrl)

  url.pathname = resolve(url.pathname, urlPath)

  if (queryString) {
    const queryParams = Object.fromEntries(new URLSearchParams(queryString))
    options.params = { ...options.params, ...queryParams }
  }

  const headers = {
    'Content-Type': 'application/json',
  }

  if (options.headers)
    Object.assign(headers, options.headers)

  // if (options.body)
  //   options.body = toQueryString(options.body)

  // if (!options.body?.timestamp && !options.params?.timestamp)
  //     options.params = { ...options.params, timestamp }

  if (signed)
    options = authenticate({ apiKey, apiSecret, options }, path)

  if (options.params)
    url.search = `?${new URLSearchParams(options.params).toString()}`

  console.log(JSON.stringify({ url, options }, null, 2))

  return fetch(url, options).then(res => res.json())
}

const init = (cfg = {}) => Object.assign(config, cfg)

export const getAssets = _ => get('v5/account/wallet-balance', {
  params: {
    accountType: 'UNIFIED',
  },
}, true)

export const getPositions = _ => get('/v5/position/list', {
  params: {
    category: 'linear',
    settleCoin: 'USDT',
  },
}, true)

const order = ({ symbol, side, quantity }) => {
  side = side.toUpperCase()
  if (side === 'BUY')
    side = 'Buy'
  else if (side === 'SELL')
    side = 'Sell'
  else
    throw new Error(`Invalid side ${side} for order.`)

  const orderParams = {
    orderType: 'Market',
    side,
    symbol,
    qty: quantity,
  }

  return post('/v5/order/create', {
    body: {
      category: 'linear',
      ...orderParams,
    },
  })
}

export default {
  platform: pf,
  name: 'Binance',
  config: init,
  init,
  request,
  get,
  post,
  put,
  delete: del,
  ws,
  order,
  getAssets,
}
