import WebSocket from 'ws'

import { platforms } from '@remora/config'

import client from './client.mjs'

import updates from './updates.mjs'

const noop = _ => {}

const pf = 'bnb'
const {
  name: platform,
  // baseUrl,
  wsUrl,
} = platforms[pf]

const defaults = {
  options: {
    url: wsUrl,
  },
  book: null,
  books: {},
  keepAlive: {
    sent: 0,
    recv: 0,
    trip: 0,
    ival: 2.5 * 1000,
    tout: 3.5 * 1000,
    timr: null,
    ping: null,
    seen: null,
    nonce: 0,
  },
}

const sockets = {}
const wsList = []

const healthcheck = sock => {
  const reportHealthcheck = msg => console.log(`HEALTHCHECK ${msg} ${platform} ${sock.options.url} ${sock.ws.url}`)
  if (sock.options.url !== sock.ws.url) {
    reportHealthcheck('URL MISMATCH')
    sock.disconnect()
    return
  }
  if (!sockets[sock.options.url]) {
    // console.log(`Socket ${sock.options.url} not registered`)
    reportHealthcheck('SOCKET NOT REGISTERED')
    sock.disconnect()
    return
  }
  if (sockets[sock.options.url] !== sock) {
    // console.log(`Socket ${sock.options.url} is ghost : reference mismatch`)
    reportHealthcheck('SOCKET REFERENCE MISMATCH')
    sock.disconnect()
    return
  }

  const ts = Date.now()
  const { recv, seen, sent, tout: timeout } = sock.keepAlive

  if (!seen || !recv) {
    reportHealthcheck('SOCKET NOT READY')
    if (sock.ws.readyState > 1) {
      console.warn(`${platform} socket not ready for healthcheck`, sock.ws.readyState)
      sock.reconnect()
    }
    return
  }

  const delta = Math.min(ts - seen, ts - recv)

  if (delta > timeout) {
    reportHealthcheck('SOCKET TIMEOUT')
    console.error({
      sou: sock.options.url,
      wsu: sock.ws.url,
      ts,
      recv,
      sent,
      seen,
      delta,
      timeout,
    })
    sock.reconnect()
  }
}
const pingSock = sock => {
  const ts = Date.now()

  sock.ws.ping()
  client.put('/fapi/v1/listenKey', true)
  sock.keepAlive.sent = ts

  sock.keepAlive.timr = setTimeout(sock.healthcheck, sock.keepAlive.tout)
}
const sockPong = sock => {
  const recv = Date.now()
  const { sent } = sock.keepAlive
  const trip = recv - sent

  sock.keepAlive = {
    ...sock.keepAlive,
    sent,
    recv,
    trip,
  }
}

const onClose = (sock, code, why) => {
  if (code === 1006)
    console.log(`${platform} Abnormal termination (r:"${why}" u:"${sock.ws?.url || sock.options.url}")`)
  else if (code === 1000)
    console.log(`${platform} Socket closed (r:"${why}" u:"${sock.ws?.url || sock.options.url}")`)
  else {
    console.warn(
      why.toString('utf8'),
      { readyState: sock.ws?.readyState },
    `Unexpected sock close : ${platform}`)
  }
}
const onError = (sock, err) => {
  console.error(err, `<- WS error (${platform})`)
  sock.reconnect()
}

const purgeBullshit = (book, _s) => {
  if (book.bids[0][0] < book.asks[0][0])
    return book

  // Assumes book is sorted + at least 1 entry in each side is valid
  const conflicts = {
    asks: book.asks.findIndex(([price]) => price > book.bids.at(-1)[0]),
    bids: book.bids.findIndex(([price]) => price < book.asks.at(-1)[0]),
  }

  Object.entries(conflicts).forEach(([side, i]) => {
    if (i === -1)
      return
    book[side] = book[side].slice(i)
  })

  return book
}

const onMessage = (sock, msg) => {
  const { e: eventType, E: ts } = msg

  if (sock.responses?.[msg.id]) {
    sock.responses[msg.id](msg)
    delete sock.responses[msg.id]
  } else if (eventType === 'depthUpdate') {
    if (ts > sock.keepAlive.seen) {
      sock.keepAlive.seen = ts

      const partial = false

      const symbol = msg.s
      const { asks, bids } = purgeBullshit({ asks: msg.a, bids: msg.b }, symbol)
      if (sock.books[symbol] && sock.books[symbol].handler) {
        const extractedData = {
          asks,
          bids,
          timestamp: ts,
          partial,
          platform,
          symbol,
        }
        sock.books[symbol].handler(extractedData)
      }
    }
  } else {
    const translate = updates.match(msg)

    if (translate) {
      const translated = translate(msg)
      const { type, data } = translated

      if (sock[type] && sock[type].listeners)
        sock[type].listeners.forEach(listener => listener(data))
      else
        console.warn(`${platform} - No listener for message ${type}`)
    } else
      console.warn(msg, `Unhandled message on ${platform}`)
  }
}

const bookSub = (sock, symbol, cb) => {
  if (!sock.books[symbol])
    sock.books[symbol] = { handler: cb }
  else if (sock.books[symbol].handler !== cb)
    sock.books[symbol].handler = cb

  const payload = {
    method: 'SUBSCRIBE',
    params: [
            `${symbol.toLowerCase()}@depth20`,
    ],
  }
  console.log(`SUB ${pf} ${symbol}`)
  sock.send(payload, _ => { console.log(`${pf} SUB ${symbol}`) })
}
const bookUnsub = (sock, symbol) => {
  const payload = {
    method: 'UNSUBSCRIBE',
    params: [
            `${symbol.toLowerCase()}@depth20`,
    ],
  }
  sock.send(payload)
  delete sock.books[symbol]
}

const auth = async (sock, credentials) => {
  return sock.reconnect({ credentials })
}
const accountSub = (sock, ev, cb) => {
  const { listenKey } = sock.options.credentials
  if (!listenKey)
    throw new Error('accountSub requires credentials / auth')

  if (!sock.account.listeners)
    sock.account.listeners = []

  if (ev === 'update') {
    sock.account.listeners.push(cb)
    // console.info(`${pf}::account:subscribed`, sock.account.listeners.length)

    sock.send({
      method: 'REQUEST',
      params: [
        `${listenKey}@account`,
        `${listenKey}@balance`,
        `${listenKey}@position`,
      ],
    }, accountInit => {
      console.info(`${pf} SUB ACCOUNT`)
      // const extractedData = extractAccountData(sock, accountInit)
      const msg = accountInit.result.reduce((data, { req, res }) => {
        if (req.endsWith('@account'))
          return { ...data, account: res }

        if (req.endsWith('@balance')) {
          return {
            ...data,
            balances: res.balances
              .filter(({ updateTime }) => Number(updateTime)),
          }
        }

        if (req.endsWith('@position')) {
          return {
            ...data,
            positions: res.positions
              .filter(({ updateTime }) => Number(updateTime)),
          }
        }

        throw new Error(`Unknown account request ${req}`)
      }, { eventType: 'ACCOUNT_INIT' })

      sock.onMessage(msg)
    })
  } else
    console.warn(`Unknown account event ${ev}`)
}
const accountUnsub = (sock, ev, cb) => {
  if (ev === 'update') {
    if (!sock.account.listeners)
      sock.account.listeners = []

    sock.account.listeners.splice(sock.account.listeners.findIndex(_cb => _cb === cb), 1)
  } else
    console.warn(`BADSUB ${pf} ${ev}`)
}

const disconnect = (sock, cb) => {
  if (!cb)
    cb = noop

  console.log(`DISCONNECT ${platform} ${sock.options.url}`)
  if (sock.ws) {
    if (sock.ws.readyState < 1) {
      sock.ws.removeAllListeners('error')
      sock.ws.on('error', noop)
      sock.ws.removeAllListeners('close')
      sock.ws.on('close', cb)
    }

    sock.ws.close()
    delete (sock.ws)
  }
  clearTimeout(sock.keepAlive.timr)
  sock.keepAlive.timr = null
  clearInterval(sock.keepAlive.ping)
  sock.keepAlive.ping = null

  sock.keepAlive.nonce = 0
}

let connectFailCount = 0
export const connect = options => {
  options = { ...defaults.options, ...options }
  let { url } = options

  const socket = sockets[url] || { ...defaults }
  if (sockets[url])
    socket.disconnect()

  socket.options = { ...socket.options, ...options }
  url = options.url

  if (options.credentials) {
    const { listenKey } = options.credentials
    if (!url.endsWith(listenKey))
      url += `/${listenKey}`
    url = new URL(url)
    if (!url.host.startsWith('fstream'))
      url.host = url.host.replace(/^.+?\./, 'fstream.')

    url = url.toString()
  }
  if (socket.options.url !== url) {
    console.log('URL', platform, url)
    socket.options.url = url
  }

  console.log(`CONNECT ${platform}`)

  socket.healthcheck = _ => healthcheck(socket)
  socket.pong = msg => sockPong(socket, msg)
  socket.ping = _ => pingSock(socket)
  socket.onError = err => onError(socket, err)
  socket.onMessage = msg => onMessage(socket, msg)
  socket.reconnect = options => {
    options = { ...socket.options, ...options }
    console.log(`CONNECT ${platform} RE`)

    return connect(options)
  }
  socket.disconnect = _ => disconnect(socket)

  socket.book = {
    on: (symbol, cb) => bookSub(socket, symbol, cb),
    off: symbol => bookUnsub(socket, symbol),
  }
  socket.account = {
    on: (ev, cb) => accountSub(socket, ev, cb),
    off: (ev, cb) => accountUnsub(socket, ev, cb),
  }

  if (connectFailCount)
    console.log(`CONNECT ${pf} ${connectFailCount}`)

  if (socket.ws)
    socket.ws.close()

  const ws = socket.ws = new WebSocket(url)
  wsList.push(ws)

  const onConnectError = err => {
    connectFailCount++

    console.error(err, `<- WS error in connect (${platform})`)
    wsList.splice(wsList.findIndex(_ws => _ws === ws), 1)

    console.error(`Connecting on ${platform} failed.`, err)
    setTimeout(_ => {
      console.log(`CONNECT ${platform} RETRY`)
      socket.reconnect()
    }, connectFailCount * 250)
  }
  ws.on('error', onConnectError)
  ws.on('message', data => socket.onMessage(JSON.parse(data)))
  ws.on('pong', _ => socket.pong())
  ws.on('close', (code, why) => {
    onClose(socket, code, why)
    wsList.splice(wsList.findIndex(_ws => _ws === ws), 1)
  })

  ws.on('open', _ => {
    ws.off('error', onConnectError)
    ws.on('error', err => {
      socket.onError(err)
    })

    connectFailCount = 0

    if (socket.keepAlive.ping)
      clearInterval(socket.keepAlive.ping)

    socket.keepAlive = {
      ...socket.keepAlive,

      sent: Date.now(),
      seen: Date.now(),
    }
    socket.keepAlive.ping = setInterval(_ => socket.ping(), socket.keepAlive.ival)

    Object.entries(socket.books).forEach(([symbol, sub]) => {
      socket.book.on(symbol, sub.handler)
    })

    console.log(`CONNECTED ${platform}`)
  })

  socket.send = (msg, onResponse) => {
    if (typeof socket.keepAlive.nonce !== 'number' || Number.isNaN(socket.keepAlive.nonce))
      console.error('NONCE NOT A NUMBER', socket.keepAlive.nonce)

    if (!socket.ws)
      return console.error('No socket in send', msg)

    if (socket.ws.readyState !== 1) {
      socket.ws.on('open', _ => {
        socket.send(msg, onResponse)
      })
      return
    }

    let id
    if (typeof msg !== 'string') {
      id = ++socket.keepAlive.nonce

      msg = JSON.stringify({
        ...msg,
        id,
      })
    }

    // UGH PLACEMENT
    socket.ws.send(msg)

    if (!id) throw new Error('No id in msg')

    if (onResponse) {
      if (socket.responses?.[id])
        throw new Error(`${platform} ws request nonce collision ${id}`)

      socket.responses = {
        ...socket.responses,
        [id]: onResponse,
      }
    }

    setTimeout(_ => {
      if (!socket.responses) {
        const fault = new Error('No responses in socket')
        console.error(socket)
        throw fault
      }
      if (socket.responses[id])
        console.error(`Response timeout ${id}`)
    }, socket.keepAlive.tout)
  }

  socket.auth = async credentials => {
    if (!credentials) {
      console.log(`AUTH ${pf} MISSING`)
      credentials = await client.post('/fapi/v1/listenKey', true)
      // console.log({ credentials })
    }

    return auth(socket, credentials)
  }

  return (sockets[url] = socket)
}

export default {
  connect,
  auth,
  // updates,
}
