import assert from 'node:assert/strict'

import { test } from 'node:test'

import client from './client.mjs'

import { writeFile } from 'node:fs/promises'

test('platforms.bnb.client:get', _t => {
  return Promise.all([

    test('platforms.bnb.client:get', async _t => {
      const ping = await client.get('/fapi/v1/ping', false)

      assert.deepStrictEqual(ping, {})
      return ping
    }),

    test('platforms.bnb.client:get:account', async _t => {
      const result = await client.get('/fapi/v2/account', true)

      const { assets: balances, positions, ...account } = result

      assert(account, 'No account data')
      assert(balances, 'No balances')
      assert(positions, 'No positions')

      // Dump for inspection
      Object.entries({ account, balances, positions }).map(([k, v]) => {
        // dump account data to file for debugging
        const dump = writeFile(`./_${k}.fish`, JSON.stringify(v, null, 2))
        return dump.then(_ => v)
      })
    }),

    test('platforms.bnb.client:get:balances', async _t => {
      const balances = await client.get('/fapi/v2/balance', true)

      assert(balances)
      assert(balances.length)

      balances.forEach(b => {
        assert(b.asset)
        assert(b.balance)
        assert(b.crossWalletBalance)
        assert(b.availableBalance)
        assert(b.maxWithdrawAmount)
      })
    }),

  ])
})

test('platforms.bnb.client:listenKey', _t => {
  return Promise.all([
    test('platforms.bnb.client:listenKey', async _t => {
      const res = await client.post('/fapi/v1/listenKey', true)

      assert(res, 'No res from post(\'/fapi/v1/listenKey\')')
      assert(res.listenKey, `res.listenKey: ${res.listenKey}`)

      const deleted = await client.delete('/fapi/v1/listenKey', {
        params: {
          listenKey: res.listenKey,
        },
      }, true)

      assert.deepStrictEqual(deleted, {})
    }),
  ])
})
