import { platforms } from '@remora/config'

import { authenticate } from './sign.mjs'

import ws from './websocket.mjs'
import { toQueryString } from '@remora/utils/queryString.mjs'

// Binance API
export const pf = 'bnb'

const config = { ...platforms[pf] }
const {
  apiKey,
  apiSecret,
  baseUrl,
} = config

// console.log(JSON.stringify(config, null, 2))

const defaults = {
  method: 'POST',
  headers: {
    contentType: 'application/json',
    'X-MBX-APIKEY': apiKey,
  },
}

export const get = (path, signed) =>
  request(path, { method: 'GET' }, signed)

export const post = (path, options, signed = true) =>
  request(path, {
    ...defaults,
    ...options,
  }, signed)

export const put = (path, options, signed = true) =>
  request(path, {
    method: 'PUT',
    ...options,
  }, signed)

export const del = (path, options, signed = true) =>
  request(path, {
    method: 'DELETE',
    ...options,
  }, signed)

export const request = async (path, options, signed = true) => {
  const timestamp = Date.now()
  options = Object.assign({}, defaults, options)

  if (typeof options === 'boolean') {
    signed = options
    options = defaults
  }

  const [urlPath, queryString] = path.split('?')
  const url = new URL(urlPath, baseUrl)

  if (queryString) {
    const queryParams = Object.fromEntries(new URLSearchParams(queryString))
    options.params = { ...options.params, ...queryParams }
  }

  const headers = {
    'Content-Type': 'application/json',
  }

  if (options.headers)
    Object.assign(headers, options.headers)

  // if (options.body)
  //   options.body = JSON.parse(options.body)

  if (!options.body?.timestamp && !options.params?.timestamp)
    options.params = { ...options.params, timestamp }

  if (signed)
    options = authenticate({ apiKey, apiSecret, ...options })

  if (options.params)
    url.search = `?${new URLSearchParams(options.params).toString()}`

  // console.log(JSON.stringify(options, null, 2))

  try {
    const response = await fetch(url, options)

    const contentType = response.headers.get('content-type')
    if (!contentType || !contentType.includes(options.headers.contentType)) {
      const document = await response.text()

      console.error(document)
      throw new Error(`Content-Type mismatch : got ${
        contentType
      } (expected ${
        options.headers['Content-Type']
      })`)
    }

    return response.json()
  } catch (err) {
    console.error(err)
  }
}

const init = (cfg = {}) => Object.assign(config, cfg)

const getAssets = () => {
  return request('api/v1/private/account/assets')
}

const order = ({ symbol, side, quantity }) => {
  const orderParams = {
    quantity,
    side: side.toUpperCase(),
    symbol,
    type: 'MARKET',
  }

  return post('fapi/v1/order', { body: toQueryString(orderParams) })
}

export default {
  platform: pf,
  name: 'Binance',
  config: init,
  init,
  getAssets,
  request,
  get,
  post,
  put,
  delete: del,
  ws,
  order,
}
