import {
  createContext, useContext,
  useEffect, useState,
} from 'react'

let ws

const connect = handlers => {
  const connectUrl = new URL('/ws/', window.location.origin)
  connectUrl.protocol = connectUrl.protocol.replace(/^http/, 'ws')

  ws = new WebSocket(connectUrl.href)

  Object.entries(handlers).forEach(([event, listener]) => {
    ws.addEventListener(event, listener)
  })

  ws.reconnect = (h = handlers) => {
    ws.close()
    connect(h)
  }
}

const register = (subs, sub) => {
  if (['on', 'off'].includes(sub.type)) {
    const { on, off, handler } = sub
    switch (sub.type) {
      case 'on':
        if (!subs[on])
          subs[on] = [handler]

        else if (!subs[on].includes(handler))
          subs[on].push(handler)

        return subs

      case 'off':
        if (!subs[off])
          return subs

        subs[off] = subs[off].filter(fn => fn !== handler)

        return subs
    }
  }

  if (sub.on) {
    const { on, handler } = sub
    if (!subs[on]) {
      return {
        ...subs,
        [on]: [handler],
      }
    }
    if (!subs[on].includes(handler)) {
      return {
        ...subs,
        [on]: [
          ...subs[on],
          handler,
        ],
      }
    }
    return subs
  }
}

const wsContext = createContext(null)
const useWebSocket = _ => {
  const ctx = useContext(wsContext)

  if (!ctx)
    throw new Error('useWebSocket must be used within a WebSocketProvider')

  return ctx
}

const SocketProvider = ({ children }) => {
  const [info, setInfo] = useState({
    sent: Date.now(),
    recv: 0,
    ping: Infinity,
    seen: 0,
    lastUpdated: 0,
  })
  const updateInfo = info => setInfo(prevInfo => ({ ...prevInfo, ...info }))

  const [subs, setSubs] = useState({})
  const updateSubs = sub => { setSubs(register(subs, sub)) }

  const onMessage = ({ type, ...data }) => {
    if (type === 'pong') {
      const T = Date.now()

      updateInfo({
        seen: T,
        recv: T,
        ping: T - data.from,
        lastUpdated: data.sent,
      })
    }

    if (type !== 'pong' && !subs[type]) {
      console.warn(`No subscribers for ${type} message.`)
      return
    }

    subs[type]?.forEach(handler => handler({ type, ...data }))
  }

  useEffect(_ => {
    const handlers = {
      open: _ => { console.info('WS open.') },
      close: _ => { console.warn('WS closed.') },
      message: e => {
        const message = JSON.parse(e.data)

        onMessage(message)
      },
      error: e => { console.error('WS error:', e) },
    }
    connect(handlers)

    const pingInterval = setInterval(_ => {
      const sent = Date.now()

      if (ws.readyState > 1) {
        console.warn(`WS not available (readyState: ${ws.readyState})`)

        setInfo(({ recv }) => {
          const ping = sent - recv
          return {
            sent,
            recv,
            ping: ping < 1000 ? ping : Infinity,
            seen: recv,
            lastUpdated: recv,
          }
        })

        ws.reconnect()
        return
      }

      if (ws.readyState === 1) {
        const message = JSON.stringify({
          type: 'ping',
          sent,
        })

        ws.send(message)

        updateInfo({
          sent,
        })
      }
    }, 1000)

    return _ => {
      clearInterval(pingInterval)
      Object.keys(handlers)
        .forEach(l => ws.removeEventListener(l, handlers[l]))

      ws.close()
    }
  }, [])

  return <wsContext.Provider value={{
    info,
    on: (type, handler) => {
      const update = {
        type: 'on',
        on: type,
        handler,
      }
      // console.warn({ sub: update })
      updateSubs(update)
    },
    off: (type, handler) => {
      const update = {
        type: 'off',
        off: type,
        handler,
      }
      updateSubs(update)
    },
  }}>
        {children}
    </wsContext.Provider>
}

export {
  SocketProvider as default, SocketProvider as Provider,
  wsContext,
  useWebSocket,
}
