import {
  createContext, useContext,
  useEffect, useState,
} from 'react'

import { fixPrice } from '@remora/utils/precision'

import { toPair, coin } from '../symbol.jsx'
import { useApi } from '../../hooks/api.mjs'

const positionsContext = createContext(null)
const usePositions = _ => {
  const ctx = useContext(positionsContext)

  if (!ctx)
    throw new Error('usePositions must be used within a PositionsProvider')

  return ctx
}

const PositionsProvider = ({ children }) => {
  const api = useApi()
  const [positions, setpositions] = useState({})

  const updatePositions = update => setpositions(prev => {
    if (!update)
      return prev

    const state = Object.entries(update).reduce((state, [symbol, position]) => {
      symbol = toPair(coin(symbol))
      // console.info({ in: 'updatePositions', symbol, position })

      return ({
        ...state,
        [symbol]: {
          ...prev[symbol],
          ...position,
          entry: fixPrice(Math.abs(position.notional / position.quantity) || 0, symbol),
        },
      })
    }, prev)

    return state
  })

  useEffect(_ => {
    api.isAuth && api.positions().then(updatePositions)
  }, [api.isAuth])

  return <positionsContext.Provider value={{ positions, updatePositions }}>
        {children}
    </positionsContext.Provider>
}

export {
  PositionsProvider as default, PositionsProvider as Provider,
  positionsContext, positionsContext as context,
  usePositions, usePositions as useProvider,
}
