import {
  createContext,
  useContext,
  useState,
  useEffect,
} from 'react'

import { buildMessage } from '@remora/utils/auth'

import { useEthereum } from '../ethereum'

const store = (key, data) => {
  if (!data) return localStorage.removeItem(key)

  if (typeof data !== 'string')
    data = JSON.stringify(data)

  localStorage.setItem(key, data)
}

const stored = k => {
  const rawData = localStorage.getItem(k)
  return JSON.parse(rawData)
}

const toHexString = data => {
  const bytes = new TextEncoder().encode(data)
  const hex = Array
    .from(bytes, b => b.toString(16).padStart(2, '0'))
    .join('')
  return `0x${hex}`
}

// TODO : Move to API ?
const validate = async payload => {
  const res = await fetch('/api/auth', {
    method: 'POST',
    headers: { 'content-type': 'application/json' },
    body: JSON.stringify(payload),
  })
  if (res.ok) {
    try {
      return res.json()
    } catch (e) {
      console.warn(e)
      return null
    }
  }
}

export const authenticate = async (eth, address) => {
  const dtf = Intl.DateTimeFormat().resolvedOptions()
  const locale = dtf.locale || dtf.locales[0] || 'en-US'
  const timezone = dtf.timeZone || 'America/New_York'

  const iat = Date.now()
  const exp = iat + 1000 * 60 * 60 * 24 * 7 // 1 week

  const msg = buildMessage({ from: address, iat, exp, lc: locale, tz: timezone })
  const hexMessage = toHexString(msg)

  const sig = await eth.request({ method: 'personal_sign', params: [hexMessage, address] })

  const sigPayload = {
    from: address,
    iat,
    exp,
    lc: locale,
    tz: timezone,
    sig,
  }

  const result = await validate(sigPayload)
  console.info({ result })
  const { cid, nic, rol } = result

  console.info({ AUTH_RESULT: result })
  return {
    key: address,
    usr: { nic, rol },
    tok: { iat, exp, cid },
  }
}

export const revoke = async ({ key: addr, tok }) => {
  store(addr, null)

  const revoked = await fetch('/api/auth', {
    method: 'DELETE',
    headers: { 'content-type': 'application/json' },
    body: JSON.stringify({ addr, tok }),
  })

  if (revoked.status === 204 || revoked.ok)
    console.info(`Revoked ${tok.cid}`)
}

const Provider = ({ children }) => {
  const { account: key } = useEthereum()

  const [auth, setAuth] = useState(stored(key))

  const updateAuth = authData => {
    console.info({ type: 'authData', authData })

    setAuth(authData)

    if (authData) {
      authData = {
        usr: authData.usr,
        tok: authData.tok,
      }
    }
    store(key, authData)
  }

  const onUpdateStorage = e => {
    if (e.key !== key) return

    updateAuth(stored(e.key))
  }

  const disconnect = _ => {
    revoke(auth)
    updateAuth(null)
  }

  useEffect(() => {
    window.addEventListener('storage', onUpdateStorage)

    if (!auth) {
      const authData = stored(key)
      // console.info({ type: 'authData', authData, for: key })
      if (authData)
        setAuth({ key, ...authData })
    }

    if (auth) {
      // console.info(auth)
      if (auth.key !== key) {
        console.info({ key, auth: auth.key })
        setAuth(stored(key))
      }

      const { exp, iat } = auth.token || {}
      const T = Date.now()
      if (T > exp || T < iat) {
        console.info({ ...auth, exp, iat })
        disconnect()
      }
    }

    return _ => window.removeEventListener('storage', onUpdateStorage)
  }, [key])

  return <AuthContext.Provider
        value={{
          auth,
          updateAuth,
          disconnect,
          challenge: authenticate,
        }}>
        {children}
    </AuthContext.Provider>
}

const AuthContext = createContext(null)
const useAuth = _ => useContext(AuthContext)

export {
  Provider as default, Provider,
  AuthContext,
  useAuth,
}
