import { useAuth } from '../providers/auth/auth.jsx'

const generateHeaders = ({ auth, params }) => {
  const headers = params.headers || {}

  if (!auth)
    return headers

  const { key, tok: { cid, exp, iat } } = auth
  // console.info({ in: 'generateHeaders', auth, params })

  return ({
    ...headers,
    'x-api-cid': cid,
    'x-api-key': key,
    'x-api-exp': exp,
    'x-api-iat': iat,
  })
}

export const authParams = params => {
  const {
    auth, ...requestParams
  } = params

  return ({
    ...requestParams,
    headers: {
      ...requestParams.headers,
      ...generateHeaders({
        params: requestParams,
        auth,
      }),
    },
  })
}

export const request = (path, params) => {
  const { headers, ...requestParams } = params

  // const base = params.baseUrl || window.location.origin
  path = '/api' + (path || requestParams.path || '/test')
  // const url = `${base}${path}`

  // console.info({
  //   type: 'request',
  //   path,
  //   base,
  //   params,
  // })

  return fetch(path, {
    method: 'GET',
    headers: {
      'content-type': 'application/json',
      ...headers,
    },
    ...requestParams,
  })
}

export const OrderParams = (order, ...params) => {
  params ||= {}

  const payload = JSON.stringify({
    type: 'order',
    order,
  })

  return {
    ...params,
    method: 'POST',
    path: '/order',
    body: payload,
  }
}

export const send = async (params = { method: 'POST' }) => {
  const { path = '/', ...requestParams } = params

  try {
    const res = await request(path, requestParams)
    if (res.ok)
      return res.json()
    else {
      const { status, statusText } = res
      const msg = `Request to ${path} failed: ${status} ${statusText}`
      try {
        const error = await res.json()
        console.warn(msg, '\n', error)
        return error
      } catch (noBody) {
        console.warn(msg, '\n', noBody)
      }
    }
  } catch (error) {
    error.message = `Error sending request to ${path}: ${error.message}`
    throw error
  }
}

export const useApi = _ => {
  const { auth } = useAuth()

  const AuthRequest = params => {
    const requestParams = authParams({ ...params, auth })

    return send(requestParams)
  }

  return {
    isAuth: !!auth,
    order: order => {
      const params = OrderParams(order)

      return AuthRequest(params)
    },
    balance: _ => AuthRequest({
      method: 'GET',
      path: '/balance',
    }),
    positions: _ => AuthRequest({
      method: 'GET',
      path: '/positions',
    }),

    request,
    authRequest: (path, params) => AuthRequest({
      ...params,
      path,
    }),
  }
}

export {
  useApi as default,
}
