import { useEffect, useState } from 'react'

import './OrderBox.css'

import {
  fixPrice,
  fixQty,
} from '@remora/utils/precision'

import { useApi } from '../../hooks/api.mjs'
// import { useAuth } from '../../providers/auth/auth.jsx'
import { useWebSocket } from '../../providers/webSocket'
import { useSymbol, toPair } from '../../providers/symbol'
import { usePrices } from '../../providers/webSocket/prices'
import { WithAuth } from '../Conditional'

import Login from '../Login'
import { useBalance } from '../../providers/auth/balance'
import { usePositions } from '../../providers/auth/positions'

const TextSpinner = ({ prefix }) => {
  const spins = prefix
    ? ['  ', '.  ', '.. ', '...', ' ..', ' ..', '  .']
    : ['|', '/', '-', '\\']
  const [spinner, setSpinner] = useState(spins[0])

  useEffect(_ => {
    const interval = setInterval(_ => {
      setSpinner(prev => {
        const index = spins.indexOf(prev) + 1
        return spins[index % spins.length]
      })
    }, 50)
    return _ => clearInterval(interval)
  }, [])

  return <span className="mono medium text-spinner">{prefix}{spinner}</span>
}

const valueDelta = (position, price) => {
  const { quantity, entry } = position
  const value = price * quantity

  return value - (entry * quantity)
}

const Position = ({ className, data, symbol, prices, trade }) => {
  if (!data || !Number(data.quantity)) return null

  const { quantity, _notional, margin, entry } = data

  const sign = quantity / Math.abs(quantity) || 0
  const side = sign > 0 ? 'long' : sign < 0 ? 'short' : ''
  const takeSide = sign > 0 ? 'bid' : sign < 0 ? 'ask' : ''
  const { [takeSide]: takePrice } = prices?.[symbol] || 0

  const closePosition = _ => trade({
    side: quantity > 0 ? 'SELL' : 'BUY',
    symbol: symbol.replace('fish', 'USDT'),
    quantity: Math.abs(quantity),
  })

  const deltaValue = valueDelta(data, takePrice)

  const positionText = `${side.toUpperCase()
    } ${Math.abs(quantity)
    } ${symbol
    } @ ${entry
    } M: ${fixPrice(margin, symbol)
    }`

  return <div className={`fullWidth space-between ${className} flex-col`}>
    <div className='mono'>{positionText}</div>
    <div className='medium nomt fullWidth flex-row space-between'>

      <div id="PNL" className='nopadding nomargin'>
        <div className={`mono lighttext ${deltaValue > 0 ? 'text-positive' : 'text-negative'}`}>
          {fixPrice(deltaValue, symbol)}
        </div>
        <div className='nomargin small'>PNL</div>
      </div>

      <div id="CLOSE" className='nopadding nomargin'>
        <button className='' onClick={closePosition}>Close</button>
      </div>

      <div id="ratio" className='nopadding nomargin'>
        <div className={`mono lighttext ${deltaValue > 0 ? 'text-positive' : 'text-negative'}`}>
          {fixPrice(100 * deltaValue / (Number(deltaValue) > 0 ? margin : margin / 2), symbol) + '%'}
        </div>
        <div className='nomargin small'>margin ratio</div>
      </div>

    </div></div>
}

export const OrderBox = () => {
  const api = useApi()

  const { symbol, coin } = useSymbol()
  const remoraSymbol = toPair(coin, 'USDT')

  const { balance, updateBalance } = useBalance()
  const { positions, updatePositions } = usePositions()

  const sortedPositions = Object.entries(positions || {})
    .sort(([s1], [s2]) => s1 === symbol ? -1 : s2 === symbol ? 1 : 0)

  const { info } = useWebSocket()
  const { prices: book } = usePrices()

  const leverage = 50

  const [ordering, setOrdering] = useState([])

  const [error, setError] = useState(null)
  const handleError = e => {
    console.error('handleError', e)
    setError(e)
  }

  const [amount, setAmount] = useState('')
  const [quantity, setQuantity] = useState('')

  const updateAmount = a => {
    if (error) {
      console.info('clearing error', error)
      setError(null)
    }
    setAmount(a)
    setQuantity('')
  }

  const updateQuantity = q => {
    if (error) {
      console.info('clearing error', error)
      setError(null)
    }
    setQuantity(q)
    setAmount('')
  }

  const { ask, bid } = book[symbol] || book[remoraSymbol] || {}
  const spread = ask - bid
  const spreadValueText = !spread
    ? 'None.'
    : fixPrice(spread, remoraSymbol).padStart(ask.length, '0')

  const sellQty = fixQty(quantity || amount * leverage / bid, remoraSymbol, Math.floor) || 0
  const sellValue = (bid * sellQty) || 0

  const buyQty = fixQty(quantity || amount * leverage / ask, remoraSymbol, Math.floor) || 0
  const buyValue = (ask * buyQty) || 0

  const formatPrice = (p, a) => fixPrice(p, remoraSymbol, a)
  const formatQty = (q, a) => fixQty(q, remoraSymbol, a)

  const buildOrder = side => {
    const order = {
      side,
      symbol: remoraSymbol,
    }
    if (quantity)
      order.quantity = quantity
    else if (amount)
      order.amount = amount

    return order
  }

  const trade = async order => {
    console.info({ in: 'OrderBox::trade', type: 'order', order })

    setOrdering(o => [
      ...o,
      order.side.toLowerCase(),
    ])
    const result = await api.order(order)
    setOrdering(o => [
      ...o.filter(o => o !== order.side.toLowerCase()),
    ])

    console.info({ in: 'OrderBox::trade', result })

    if (result.error) {
      handleError(result.error)
      return
    }

    if (result.balance)
      updateBalance(result.balance)
    if (result.positions)
      updatePositions(result.positions)
  }

  const positionsStats = Object.entries(positions || {})
    .reduce((stats, [symbol, position]) => {
      const closeSide = position.quantity < 0 ? 'ask' : 'bid'
      const closePrice = book[symbol]?.[closeSide]
      const fallbackPrice = closeSide === 'ask' ? Infinity : 0

      const usedMargin = +position.margin + stats.usedMargin || 0
      const unrealised = stats.unrealised + valueDelta(position, closePrice || fallbackPrice)

      return {
        ...stats,
        usedMargin,
        unrealised,
      }
    }, {
      usedMargin: 0,
      unrealised: 0,
    })

  const { usedMargin, unrealised } = positionsStats
  const profitClass = !unrealised
    ? 'faded'
    : unrealised > 0
      ? 'text-positive'
      : 'text-negative'

  return (<div className='content flex-col smallms fullHeight'>
    <div className="box">
      <div className='box__item'>
        <h2>{symbol}</h2>
      </div>

      <div id='price-ask' className="box__item nomb">
        <label htmlFor="askField">Ask</label>
        <input
          name="ask"
          id="askField"
          value={ask || ''}
          type="number"
          className="box__item__value mono"
          disabled />
      </div>

      <div id='price-spread' className="box__item smallmt smallmb">
        <label htmlFor="spreadField" className='faded'>Spread</label>
        <input
          name="spread"
          id="spreadField"
          value={spreadValueText}
          // type="number"
          className="box__item__value mono"
          disabled />
      </div>

      <div id='price-bid' className="box__item nomt">
        <label htmlFor="bidField">Bid</label>
        <input
          name="bid"
          id="bidField"
          value={bid || ''}
          type="number"
          className="box__item__value mono"
          disabled />
      </div>

      <div className={`box__item flex-col nomb nomt ${error ? 'border-alert' : ''}`}>
        <div id='order-amount' className="box__item smallmb">
          <label htmlFor="amountField">$fish</label>
          <input
            name="amount"
            id="amountField"
            value={amount}
            type="number"
            className={`box__item__value mono ${amount ? '' : 'faded'}`}
            onChange={e => { updateAmount(e.target.value) }}
            placeholder={amount ? '' : balance ? `Max: ${fixPrice(balance.margin, symbol, Math.floor)}` : 'No balance.'}
            disabled={!balance} />
        </div>

        <div id='order-quantity' className="box__item smallmt">
          <label htmlFor="quantityField">${coin}</label>
          <input
            name="quantity"
            id="quantityField"
            value={quantity}
            type="number"
            className={`box__item__value mono ${quantity ? '' : 'faded'}`}
            onChange={e => { updateQuantity(e.target.value) }}
            disabled={!balance} />
        </div>

        <div id='socket' className="box__item box__info">
          <div>{info.lastUpdated
            ? <>Last updated <span className="mono medium"> {
              `${new Date(info.lastUpdated).toLocaleTimeString()}`} </span> </>
            : <TextSpinner prefix='Connecting' />}
          </div>
          <div>
            <span>
              {info.ping && info.ping < 1000
                ? <>
                  <span className="mono medium">{`${info.ping} ms`}</span> latency
                </>
                : <span>Not connected.</span>
              }
            </span>
          </div>
        </div>

        <div id='actions' className='box__item mono nomb'>
          <WithAuth fallback={<Login className="box__button fullWidth" />}>
            <button name="buy"
                    onClick={_ => trade(buildOrder('BUY'))}
                    className="box__button"
                    disabled={!Number(buyQty) || ordering.includes('buy')}>
              Buy<br />
              {
                formatQty(buyQty, Math.floor)} (${
                formatPrice(buyValue).padStart(amount.length + 3)})<br />M: {
                formatPrice(buyValue / leverage, Math.ceil)
            }</button>

            <button name="sell"
                    onClick={_ => trade(buildOrder('SELL'))}
                    className="box__button"
                    disabled={!Number(sellQty) || ordering.includes('sell')}>
              Sell<br />
              {
                formatQty(sellQty, Math.floor)} (${
                formatPrice(sellValue).padStart(amount.length + 3)})<br />M: {
                formatPrice(sellValue / leverage, Math.ceil)}
            </button>
          </WithAuth>
        </div>

        <span className="mono medium text-alert">
          {error ? error.message : '  '}
        </span>
      </div>

      <WithAuth>
        {balance?.margin &&
          <div id='balance' className="box__item box__info">
            <div className='medium nomb rightText'>
              <div>
                <span>Total: </span>
                <span className={`mono ${profitClass}`}>
                  {formatPrice(+balance.margin + usedMargin + unrealised)}
                </span>
              </div>

              <div>
                <span>Margin: </span>
                <span className={'mono'}>
                  {formatPrice(balance.margin)}
                </span>
              </div>
            </div>
            <div className='medium nomb'>
              <div>
                <span className={`mono ${profitClass}`}>
                  {formatPrice(unrealised)}</span>

                <span> profit</span>
              </div>

              <div>
                <span className='mono faded'>
                  {formatPrice(usedMargin)}</span>
                <span> locked</span>
              </div>
            </div>
          </div>}

        {sortedPositions && sortedPositions.map(([symbol, position]) => {
          return <Position key={symbol} className='box__item' data={position} symbol={symbol} prices={book} trade={trade} />
        })}
      </WithAuth>
    </div>
  </div>)
}

export default OrderBox
