import { useState } from 'react'

import { useEthereum } from '../providers/ethereum'
import { useAuth } from '../providers/auth/auth'

import Collapsible from './Collapsible.jsx'
import Conditional from './Conditional.jsx'

export const ConnectButton = ({ handleConnect }) => {
  if (!handleConnect)
    handleConnect = useEthereum().connect

  return <button onClick={handleConnect}>
        Connect with Ethereum
    </button>
}
const NoEth = () => <div className='flex fullWidth fullHeight centerContent centerItems'>
    <p className='centerText smallms'>
        No Ethereum connection detected.<br />
        Install a wallet such as Metamask, or use a browser with Ethereum support.
    </p>
</div>

const AuthButton = ({ runAuth: authenticate, onComplete: fulfill }) => {
  const [statusText, setStatus] = useState('Authenticate')

  const disabled = ['Signing...', 'Validating...', 'Success!'].includes(statusText)

  const handleAuth = async () => {
    setStatus('Signing...')
    try {
      setStatus('Validating...')
      const authData = await authenticate()
      if (!authData) throw new Error('Could not authenticate.')

      setStatus('Success!')
      console.info(authData, '<- authenticated')
      setTimeout(_ => {
        fulfill(authData)
      }, 900)
    } catch (e) {
      console.error(e)
      setStatus(e.message)
    }
  }

  return <button
        onClick={handleAuth}
        disabled={disabled}>
        {statusText}
    </button>
}

const Login = ({ className }) => {
  // console.info('Init connect')
  const { account, eth, connect } = useEthereum()
  const { auth, updateAuth, challenge, disconnect } = useAuth(account)

  // console.info({ account, auth }, '<- In Login')

  return <div className={`fullHeight flex-col centerItems ${className}`}>
        <Conditional check={eth} fallback={<NoEth />}>

            <Conditional check={account} fallback={<ConnectButton handleConnect={connect} />}>

                <Conditional check={auth?.usr} fallback={<AuthButton
                    runAuth={() => challenge(eth, account)}
                    onComplete={updateAuth} />}>

                    <button onClick={_ => disconnect()}>Disconnect</button>
                    <TokenInfo data={auth} revoke={disconnect} />

                </Conditional>
            </Conditional>
        </Conditional>

    </div>
}

const DateDisplay = ({ ts, formatter }) => {
  const date = new Date(ts)
  const formatted = formatter
    ? formatter(date)
    : date.toLocaleString()

  return <span className='mono'>{formatted}</span>
}

const TokenInfo = ({ data, _revoke }) => {
  const { usr: user, tok: { iat, exp } } = data

  return <div className='flex flex-col'>
        <h2 className='nomargin'>Connected as {user.rol[0] || 'guest'}</h2>
        <Collapsible title='Token info'>
            <p className='flex fullWidth nomargin space-between smallText'><span>Since: </span><DateDisplay ts={iat} /></p>
            <p className='flex fullWidth nomargin space-between smallText'><span>Until: </span><DateDisplay ts={exp} /></p>
        </Collapsible>
    </div>
}

export default Login
