import { MongoClient as dbClient } from 'mongodb'

const {
  MONGO_HOST = 'battlestation.local',
  MONGO_PORT = '27017',
  MONGO_DBNAME = 'test',
} = process.env

const books = {
  remora: {
    BTCUSDT: {
      asks: [],
      bids: [],
    },
    ETHUSDT: {
      asks: [],
      bids: [],
    },
  },
}

const users = {
  '0x7cd974d9ce1014a3feb5f359c5d9b83174209782': {
    uid: 0x00,
    nic: 'epsilon',
    rol: 'owner',
  },
  '0xde4f75ea20f2d623ee0a820b48786af30a9010c2': {
    uid: 0x01,
    nic: 'LittlePotato',
  },
  '0xc7e4be4949f479f5044c2e377661ca7245aba590': {
    uid: 0x02,
    rol: 'test',
  },
  '0xbe1f98c407e7030904fdac86219f5c9fb7029ecf': {
    uid: 0x03,
    nic: 'sequoya.eth',
  },
}

const balances = {
  remora: {
    krk: {
      total: 0n,
      long: 0n,
      short: 0n,
    },
  },
  '0x7cd974d9ce1014a3feb5f359c5d9b83174209782': {
    margin: 1000,
  },
  '0xde4f75ea20f2d623ee0a820b48786af30a9010c2': {
    margin: 1000,
  },
  '0xc7e4be4949f479f5044c2e377661ca7245aba590': {
    margin: 1000,
  },
  '0xbe1f98c407e7030904fdac86219f5c9fb7029ecf': {
    margin: 1000,
  },
}

const defaults = {
  users,
  balances,
  positions: {},
  books,
}

const URL = `mongodb://${MONGO_HOST}:${MONGO_PORT}/${MONGO_DBNAME}`

console.log(`connecting to ${URL}`)

const connected = dbClient.connect(URL, {
  connectTimeoutMS: 1000, // Default is 30s :(
  family: 4, // Force IPV4 because practical reasons
})

const dbInit = connected.then(connection => {
  console.log(`connected to ${URL}`)

  const db = connection.db(MONGO_DBNAME)
  console.log(`Got DB ${MONGO_DBNAME}`)

  const initResource = async (resource, fallback) => {
    const collection = db.collection(resource)
    const keys = Object.keys(fallback)

    const updates = keys.map(key => ({
      updateOne: {
        filter: { _id: key },
        update: {
          $set: { _id: key },
          $setOnInsert: fallback[key],
        },
        upsert: true,
      },
    }))

    if (!updates.length)
      return [resource, fallback]

    const initialised = collection.bulkWrite(updates, { ordered: false })
      .then(_ => collection.find().toArray())
      .then(arr => arr.reduce((acc, { _id: key, ...doc }) => ({
        ...acc, [key]: doc,
      }), {}))

    return [resource, await initialised]
  }

  const initialised = Promise.all(Object.entries(defaults).map(([resource, fallback]) =>
    initResource(resource, fallback),
  ))

  return initialised.then(Object.fromEntries)
})

const state = await dbInit

console.log(state)
