import DataStore from '../state/index.mjs'

import {
  // fixNumber,
  fixPrice,
} from '@remora/utils/precision.mjs'

import // clients,
{
  bnb as Binance,
  krk as Kraken,
  byb as Bybit,
} from '@remora/platforms'

import { applyFees } from '../state/books.mjs'
import { persist } from '../data/db.mjs'
import { computeRouting } from '../state/routing.mjs'

const sockets = {
  bnb: Binance.ws.connect(),
  krk: Kraken.ws.connect(),
  byb: Bybit.ws.connect(),
}

const symbols = ['ETHUSDT', 'BTCUSDT']

let routing = {}

export const getRouting = (symbol, priceSort) => {
  // TODO : This is ugg asf
  const bidSort = ({ bid: { offset: b1 } }, { bid: { offset: b2 } }) =>
    Number(b2) - Number(b1)
  const byBid = ['sell', 'short', 'bid', 'bids']
  const askSort = ({ ask: { offset: a1 } }, { ask: { offset: a2 } }) =>
    Number(a1) - Number(a2)
  const byAsk = ['buy', 'long', 'ask', 'asks']

  if (symbol) {
    if (priceSort) {
      const compare = byAsk.includes(priceSort)
        ? askSort
        : byBid.includes(priceSort)
          ? bidSort
          : _ => {}

      return routing[symbol].sort(compare)
    }

    return routing[symbol]
  }

  if (Object.keys(routing).length)
    return routing
}

const userPrices = {}
export const latestPrices = symbol => {
  if (!Object.keys(routing).length)
    return

  const updateTime = Date.now()
  const prices = Object.entries(routing).reduce((acc, [symbol, routes]) => {
    const { long, short } = routes.reduce((best, route) => {
      const {
        ask: { offset: ask },
        bid: { offset: bid },
      } = route

      if (Number(best.long.ask.offset) > Number(ask))
        best.long = route
      if (Number(best.short.bid.offset) < Number(bid))
        best.short = route

      return best
    }, { long: routes[0], short: routes.at(-1) })

    let { ask: { offset: ask } } = long
    let { bid: { offset: bid } } = short
    const spread = ask - bid

    if (spread < 0) {
      // TODO - better coefficient ?
      const offset = spread / 2

      ask = +ask - offset
      bid = +bid + offset
    }

    const cached = userPrices[symbol]
    const update = {
      ask: fixPrice(ask, symbol, Math.ceil),
      bid: fixPrice(bid, symbol, Math.floor),
      T: updateTime,
    }
    if (cached?.ask !== update.ask || cached?.bid !== update.bid)
      userPrices[symbol] = { ...update }

    return {
      ...acc,
      [symbol]: update,
    }
  }, {})

  if (symbol)
    return prices[symbol]

  return prices
}

const onBook = (pf, symbol, book) => {
  if (!book)
    throw new Error('no book in update ')

  // Note : remora book no longer returning partial updates
  const books = DataStore.updateBooks(pf, symbol, book)

  const pfCount = Object.keys(sockets).length
  const booksReady = books => {
    if (!books)
      return

    for (const symbol of symbols) {
      if (!books[symbol])
        return false

      const pair = books[symbol]
      for (const side of ['asks', 'bids']) {
        if (pair[side].length !== pfCount) {
          // console.warn(pair, side, `Not ready because ${symbol}`)
          return false
        }
      }
    }

    return true
  }
  const accountsReady = accounts => {
    return (Object.keys(accounts).length === pfCount)
  }

  if (booksReady(books)) {
    const accounts = DataStore.getAccounts()
    if (!accountsReady(accounts))
      return

    routing = computeRouting(books, accounts)

    const prevPrices = { ...userPrices }
    const updated = Object
      .entries(latestPrices())
      .filter(([symbol, prices]) => {
        const prev = prevPrices[symbol]
        if (!prev) return true

        for (const k of ['ask', 'bid']) {
          if (prev[k] !== prices[k])
            return true
        }

        return false
      })
    if (updated.length) {
      const update = {
        type: 'prices',
        prices: Object.fromEntries(updated),
      }

      notifyListeners(update)
    }
  }
}

const onAccountUpdate = (pf, update) => {
  if (!update)
    throw new Error(`Can't update ${pf} with ${update}`)

  DataStore.platforms.update(pf, update)
}
Object.entries(sockets).forEach(([pf, sock]) => {
  symbols.forEach(symbol => {
    sock.book.on(symbol, book => onBook(pf, symbol, book))
  })

  if (sock.auth) {
    console.log(`${pf}::socket:auth`)
    sock.auth?.().then(_ => {
      if (sock.account)
        sock.account.on('update', data => onAccountUpdate(pf, data))
      else {
        console.warn(`${pf}::account:unavailable`)
        sock.reconnect()
      }
    })
  }
})

const updateListeners = []
const notifyListeners = update => updateListeners.forEach(handler => handler(update))

export const subscribe = listener => {
  updateListeners.push(listener)

  return _ => updateListeners
    .splice(updateListeners.findIndex(l => l === listener))
}

const externals = _ => {
  const epoch = Date.now()
  const books = DataStore.getBooks()
  const pfs = Object.keys(books).filter(pf => pf !== 'remora')

  const externals = pfs.reduce((platforms, pfName) => {
    const platform = books[pfName]
    const bestPrices = Object.entries(platform)
      .sort(([s1], [s2]) => s2 < s1 ? -1 : 1).reduce(
        (symbols, [symbol, {
          asks,
          bids,
          timestamp,
        }]) => {
          const bestBid = bids?.[0]?.[0]
          const bestAsk = asks?.[0]?.[0]
          const bidCost = applyFees({
            price: bestBid,
            pf: pfName,
            side: 'bids',
          })
          const askCost = applyFees({
            price: bestAsk,
            pf: pfName,
            side: 'asks',
          })

          return {
            ...symbols,
            [symbol]: {
              ask: askCost,
              bid: bidCost,
              askRaw: bestAsk,
              bidRaw: bestBid,
              dat: timestamp,
              age: epoch - timestamp,
            },
          }
        }, {})

    return {
      ...platforms,
      [pfName]: bestPrices,
    }
  }, { remora: latestPrices() })

  return externals
}

setInterval(_ => {
  persist('prices', externals())
    .then()
}, 1000)

export default {
  subscribe,
  latest: latestPrices,
  routing: getRouting,
  externals,
}
