import assert from 'node:assert'

import { fixQty, fixQuote } from '@remora/utils/precision.mjs'
import platforms from '@remora/platforms'

// import seq from '../sequence.mjs'

import { getPrices } from '../../state/books.mjs'
import { getRouting, latestPrices } from '../../feed/feed.mjs'
import { clearPosition, patch as patchPositions } from '../../state/positions.mjs'
import { patch as patchBalances } from '../../state/balances.mjs'
import { dump } from '@remora/utils'

// TODO : Move to platforms or something IDK ?
const minNotional = {
  BTCUSDT: 100,
  ETHUSDT: 20,
}

const userSymbols = {
  BTCUSDT: 'BTCfish',
  ETHUSDT: 'ETHfish',
}
const toUserSymbol = s => userSymbols[s] || s

const signof = v => {
  if (['buy', 'sell'].includes(v))
    return v === 'buy' ? 1 : -1

  return v / Math.abs(v) || 0
}

// builder for action { type: order, order: { ... } }
export const order = action => {
  assert.strictEqual(typeof action, 'object')
  assert.strictEqual(action.type, 'order')

  try {
    assert.strictEqual(typeof action.order, 'object')
  } catch (e) {
    console.log(JSON.stringify(action, null, 2), '---------------------------------')
  }
  assert(action.order.symbol && action.order.side)
  assert(['buy', 'sell'].includes(action.order.side))

  assert(action.order.quantity || action.order.amount, 'Invalid order: no quantity or amount')
  assert(action.order.quantity
    ? action.order.quantity > 0
    : action.order.amount > 0, 'Invalid order: negative quantity or amount')

  const { symbol, side } = action.order
  const sign = signof(side)
  const take = side === 'buy'
    ? 'ask'
    : 'bid'
  const leverage = action.leverage || 50

  const order = {
    from: action.from || action.order.from,
    symbol,
    sign,
    side,
    take,
    leverage,
  }

  if (action.order.quantity)
    order.quantity = action.order.quantity
  else
    order.amount = action.order.amount

  if (action.order.price)
    order.price = action.order.price
  else {
    const prices = latestPrices(symbol)
    const { [take]: userPrice } = prices
    console.log(symbol, userPrice, latestPrices(symbol))
    const remoraPrice = getPrices('remora', symbol)[take]
    order.price = userPrice

    // TODO : Kraken shit somewhere else ?
    // const tolerance = 0.01
    // assert(sign * order.price > sign * remoraPrice + sign * tolerance,
    //         `Invalid ${side} price (u:${order.price} ${sign > 0 ? '<' : '>'} r:${remoraPrice})`)
  }

  order.quantity = fixQty(action.order.quantity || (action.order.amount * leverage / order.price),
    symbol,
    Math.floor)

  const validate = state => {
    // const remoraPrices = getPrices('remora', symbol)
    // const remoraPrice = remoraPrices[take]

    const { symbol, price } = order

    // console.log({
    // remoraPrices,
    // userPrices,
    // remoraPrice,
    // orderPrice: order.price,
    // price,
    // }, 'in order builder')

    const quantity = fixQty(order.quantity || (order.amount * leverage / price),
      symbol,
      Math.floor)
    const quote = fixQuote(quantity * price, symbol, Math.floor)

    const userSymbol = toUserSymbol(symbol)
    try {
      const { [symbol]: minQuote } = minNotional

      assert(Number(quote) > (minQuote || 10), `${userSymbol} order too small. (${quote} < ${minQuote})`)
    } catch ({ message: orderTooSmall }) {
      const e = new Error(orderTooSmall)

      e.detail = {
        symbol: userSymbol,
        quote,
        minimum: minNotional[order.symbol],
      }

      throw e
    }

    // console.log({
    //   action,
    //   order,
    //   quantity,
    // }, 'in order builder')

    const balance = state.balances[order.from]
    assert(balance)

    const position = state.positions[order.from]?.[symbol] || {
      ...clearPosition(order.from, symbol),
    }

    // console.log({
    //   balance,
    //   position,
    // }, 'in order builder')

    assert.strictEqual(typeof position, 'object')
    assert.strictEqual(typeof balance, 'object')

    const positionSign = signof(position.quantity)
    const positionQuantity = Math.abs(position.quantity)

    // console.log({
    //   positionSign,
    //   positionQuantity,
    // }, 'in order builder')

    const closingQty = positionSign !== sign
      ? Math.min(positionQuantity, quantity)
      : 0

    const closingValue = closingQty && closingQty * price
    const closingNotional = closingQty && closingQty * position.notional / positionQuantity
    const closingMargin = fixQuote(closingNotional / leverage, symbol, Math.floor)

    const delta = closingNotional - closingValue
    const pnl = fixQuote(delta * sign, symbol, delta * sign < 0 ? Math.floor : Math.ceil)

    const returnedMargin = +closingMargin + +pnl

    const openingQty = quantity - closingQty
    const openingQuote = fixQuote(openingQty * price, symbol, sign > 0 ? Math.ceil : Math.floor)
    const marginCost = fixQuote(openingQuote / leverage, symbol, Math.ceil)

    try {
      assert(Number(balance.margin) >= Number(marginCost), 'Insufficient margin')
    } catch ({ message: insufficientMargin }) {
      const e = new Error(insufficientMargin)

      e.detail = {
        required: marginCost,
        available: balance.margin,
      }

      throw e
    }

    const diff = {
      balances: {
        [order.from]: {
          margin: fixQuote(returnedMargin - marginCost, symbol, Math.floor),
        },
      },
      positions: {
        [order.from]: {
          [symbol]: {
            quantity: fixQty(quantity * sign, symbol),
            notional: fixQuote(openingQuote - closingNotional, symbol),
            margin: fixQuote(marginCost - closingMargin, symbol),
          },
        },
      },
    }

    return diff
  }

  const run = async state => {
    const diff = validate(state)
    const direction = { sell: 'short', buy: 'long' }[order.side]
    if (!direction)
      throw new Error('wat')

    const routes = getRouting(order.symbol, take)
    const route = routes.find(r => r.canOrder[direction] > (order.quantity * order.price / order.leverage))
    if (!route)
      throw new Error('Routing error.')

    const client = platforms[route.pf]

    const orderParams = { ...order, price: route[take].raw }
    const result = await client.order(orderParams)
    console.log(JSON.stringify({
      result,
    }, null, 2), route.src)

    // crashDump({ route, orderParams })
    dump({ route, orderParams, pf: route.pf, result })

    return diff
  }

  return {
    action: { type: 'order', order },
    validate,
    run,
    // diff: validate,
    apply: state => {
      // const { [seq]: cs } = state
      // assert(cs, 'Invalid state : no sequence')
      const diff = validate(state)

      const next = {
        balances: patchBalances(diff.balances, state.balances),
        positions: patchPositions(diff.positions, state.positions),
      }

      return next
    },
  }
}

export default order

// TODO : Rewrite sequence stuff :-)
