import assert from 'node:assert'

import { seq } from '../sequence.mjs'
import { hash } from '../hash.mjs'

export const builder = action => {
  const validate = state => {
    assert.strictEqual(action.type, 'increment')
    assert.strictEqual(typeof action.increment, 'string')

    const current = state[action.increment] ?? 0
    assert.strictEqual(typeof (1 + current), 'number',
            `State has incompatible value [${action.increment}]: ${current}`)

    const v = current + 1
    assert.strictEqual(typeof v, 'number')

    return { [action.increment]: v }
  }

  const diff = state => {
    const prev = state[action.increment] ?? 0
    assert.strictEqual(typeof prev, 'number')

    const next = prev + 1
    const diff = next - prev // Shaped like the original value

    assert.strictEqual(typeof diff, 'number',
            `Invalid diff ${diff} for ${prev} -> ${next}`)

    // Return next state.
    return { [action.increment]: next }
  }

  const apply = state => {
    // Validate state.
    const changes = validate(state)
    assert(state[seq], 'Invalid state : no sequence')

    // Apply sequence.
    const {
      [seq]: cs,
      ...data
    } = state

    const next = {
      ...data,
      ...changes,
    }
    const ns = {
      s: cs.s + 1,
      h: hash(cs.h, changes),
      c: hash(next),
    }

    return {
      [seq]: ns,
      ...next,
    }
  }

  return { // describe 'increment' sequence
    validate,
    apply,
    diff,
  }
}

export default builder
