import EventEmitter from 'node:events'

import { subscribe as onPrices } from '../feed/feed.mjs'
import { fixQuote } from '@remora/utils'
import { getPositions } from '../state/positions.mjs'

const liquidator = new EventEmitter()

const infer = (position, price) => {
  const { leverage = 50, quantity, margin, notional } = position

  const entry = Math.abs(notional / quantity)
  const maint = (notional / leverage) / 2 // Trigger liq @50%
  const pnl = (price - entry) * quantity
  const liquidity = +margin + pnl

  return ({
    entry,
    maint,
    pnl,
    liquidity,
  })
}

onPrices(({ prices }) => {
  const positionsLiquidity = Object.entries(getPositions())
    .flatMap(([wallet, positions]) => Object
      .entries(positions)
      .filter(([symbol, position]) => prices[symbol] && +position.quantity)
      .map(([symbol, position]) => {
        const closeSide = Number(position.quantity) > 0
          ? 'bid'
          : 'ask'
        const closePrice = prices[symbol][closeSide]
        const status = infer(position, closePrice)

        const lossToLiq = (status.liquidity - status.maint) / status.maint

        const result = {
          wallet,
          symbol,
          position,
          status: {
            price: closePrice,
            ...status,
            liquidity: fixQuote(lossToLiq, symbol, Math.floor),
            pnl: fixQuote(status.pnl, symbol, Math.floor),
            illiquid: +status.liquidity < status.maint,
          },
        }

        return result
      }))

  const toLiquidate = positionsLiquidity
    .filter(({ status }) => status.illiquid)

  if (toLiquidate.length)
    liquidator.emit('illiquid', toLiquidate)
})

export default liquidator
