import { MongoClient } from 'mongodb'

import { into } from './persist.mjs'

const IP_VERSION = 4 // TODO config

const {
  DB_PROTOCOL = 'mongodb',
  MONGO_HOST: DB_HOST = 'battlestation.local',
  MONGO_PORT: DB_PORT = '27017',
  DB_DBNAME = 'test', // process.env.DB_ENV || 'test', // TODO : DB_ENV
} = process.env

const DB_URL = `${DB_PROTOCOL}://${DB_HOST}:${DB_PORT}/${DB_DBNAME}`

const client = new MongoClient(DB_URL, {
  connectTimeoutMS: 1000, // Default 30s
  serverSelectionTimeoutMS: 1000, // Default 30s
  family: IP_VERSION, // Force IPV4 because practical reasons
})

const connect = _ => {
  console.log(`connecting to ${DB_URL}`)

  return client.connect()
}
const connected = connect()

const db = connected.then(_ => {
  console.log(`connected to ${DB_URL}`)

  return client.db(DB_DBNAME)
})

// const init = async (defaultState = defaults) => {
const init = async defaultState => {
  const client = await connected
  if (!client)
    return defaultState

  const session = client.startSession()
  const db = client.db(DB_DBNAME)

  const initResource = async (resource, fallback, session) => {
    const collection = db.collection(resource)
    const keys = Object.keys(fallback)

    const updates = keys.map(key => ({
      updateOne: {
        filter: { _id: key },
        update: {
          $set: { _id: key },
          $setOnInsert: fallback[key],
        },
        upsert: true,
      },
    }))

    const applyDefaults = !updates.length
      ? Promise.resolve()
      : collection.bulkWrite(updates, { session, ordered: false })

    const initialised = applyDefaults
      .then(_ => collection.find().toArray())
      .then(arr => arr.reduce((acc, { _id: key, ...doc }) => ({
        ...acc, [key]: doc,
      }), {}))

    return [resource, await initialised]
  }

  const initialised = Promise.all(Object.entries(defaultState).map(([resource, fallback]) =>
    initResource(resource, fallback, session),
  ))

  return initialised.then(Object.fromEntries).then(async state => {
    await session.endSession()

    return state
  }).catch(async err => {
    await session.abortTransaction()
    await session.endSession()

    console.err(err, 'Failed to initialise state from DB')
  })
}

const persist = async (set, data, id) => {
  const persisted = into(await db, set, data, id)
  return persisted
}

export {
  client, connect,
  db as default, db,
  init as initState,
  persist,
}
