import { api as config } from '@remora/config/config.mjs'

import Fastify from 'fastify'
import cors from '@fastify/cors'
import websocket from '@fastify/websocket'

import { randomUUID as UUID } from 'node:crypto'

import { formatBigEntries } from '@remora/utils/precision.mjs'

import DataStore from './state/index.mjs'
import engine from './engine/index.mjs'

import {
  validateSignature,
  authorize,
} from './auth.mjs'

import priceFeed from './feed/feed.mjs'
import { hasOneOfRoles } from './state/users.mjs'

const api = Fastify({
  logger: config.logger,
  trustProxy: config.trustProxy,
})
const sessions = {}

api.register(cors, { origin: '*' })

api.register(websocket, { options: { clientTracking: true, maxPayload: 1048576 } })
api.register(async api => {
  const { websocketServer } = api
  const unsubscribe = priceFeed.subscribe(update => {
    const payload = JSON.stringify(update)
    websocketServer.clients.forEach(client => client.send(payload))
  })
  websocketServer.on('close', unsubscribe)

  api.get('/', { websocket: true }, (connection, _req) => {
    const { socket } = connection

    // console.log(connection)
    // console.log(socket)

    const latestPrices = priceFeed.latest()
    if (latestPrices) {
      const update = {
        type: 'prices',
        prices: latestPrices,
      }
      socket.send(JSON.stringify(update))
    }

    // Web stuff such as auth and whatever ?
    socket.on('message', msg => {
      try {
        const message = JSON.parse(msg)
        if (message.type === 'ping' && Number(message.sent)) {
          connection.socket.send(JSON.stringify({
            type: 'pong',
            prices: priceFeed.latest(),
            from: message.sent,
            sent: Date.now(),
          }))
        } else console.info(message)
      } catch (err) {
        console.error(err)
      }
    })
  })
})

api.post('/auth', async (req, res) => {
  // TODO : Fastify schema
  console.info(req.body, '<- got auth request')

  const time = Date.now()
  if (time > req.body?.exp || time < req.body?.iat)
    res.status(400).send()

  try {
    const isValid = validateSignature(req.body)
    if (!isValid)
      return res.status(401).send()
  } catch (err) {
    console.error(err)
    return res.status(400).send()
  }

  const { from: address, sig, lc, tz } = req.body

  // TODO : https://redis.io/commands/expire/ ?
  const cid = UUID()
  sessions[cid] = { sig, lc, tz }

  const usr = DataStore.getUser(address)
  const rol = DataStore.getRoles(address)
  const connected = {
    cid,
    nic: usr?.nic,
    rol,
  }
  try {
    connected.bal = formatBigEntries(DataStore.balances.get(address))
    console.log({ connected })
  } catch (err) {
    console.error(err)
  }

  console.log(connected, '<- connected')

  return res.header('Access-Control-Allow-Origin', '*')
    .header('Content-Type', 'application/json')
    .send(connected)
})

api.delete('/auth', async (req, res) => {
  console.log({
    revoke: {
      addr: req.body.addr,
      tok: req.body.tok,
    },
  })
  try {
    const { addr, tok } = req.body
    const { iat, exp, cid } = tok
    if (!sessions[cid])
      return res.status(204).send()
    const { sig, lc, tz } = sessions[cid]

    const isValid = validateSignature({ from: addr, sig, iat, exp, lc, tz })
    if (isValid) {
      delete sessions[cid]
      return res.status(200).send()
    } else
      return res.status(401).send()
  } catch (err) {
    console.error(err)
    return res.status(400).send()
  }
})

const validateOrder = ({ type, ...params }) => {
  if (!params[type])
    throw new Error('Invalid order type.')

  console.log({
    type: 'validate',
    input: type,
    from: params.from,
    [type]: params[type],
  })

  if (params.order) {
    try {
      const {
        side,
        symbol,
      } = params[type]

      const validated = {
        type,
        side: side.toLowerCase(),
        symbol,
      }

      if (params[type].quantity)
        validated.quantity = params[type].quantity
      else if (params[type].amount)
        validated.amount = params[type].amount
      else
        throw new Error('Invalid quantity or amount.')

      return validated
    } catch ({ message }) {
      const err = new Error(message)
      throw err
    }
  }
}

api.post('/order', async (req, res) => {
  console.log({
    headers: req.headers,
    body: req.body,
  })

  const { key } = authorize(req)

  if (!key)
    return res.status(401).send()

  try {
    const { type, order: params } = req.body

    if (type !== 'order')
      return res.status(400).send('Invalid type.')

    if (!params)
      return res.status(400).send('No order.')

    const validated = validateOrder({
      type: 'order',
      order: params,
      from: key,
    })

    console.log({ validated }, 'validated order')

    if (!validated)
      return res.status(400).send('Invalid order.')

    try {
      const action = {
        type: 'order',
        order: validated,
        from: key,
      }
      const result = await engine.execute(action)

      const userUpdate = {}

      if (result.balances?.[key])
        userUpdate.balance = result.balances[key]

      if (result.positions?.[key])
        userUpdate.positions = result.positions[key]

      return res.status(200).send(userUpdate)
    } catch (err) {
      console.error({ err, order: validated })
      const sentError = {}
      if (err.message)
        sentError.message = err.message
      if (err.detail)
        sentError.detail = err.detail

      return res.status(400).send({
        result: 'error',
        error: sentError,
        data: 'order',
        order: validated,
      })
    }
  } catch (err) {
    console.error(err)

    const sentError = {}
    if (err.message)
      sentError.message = err.message
    if (err.detail)
      sentError.detail = err.detail

    return res.status(400).send({
      result: 'error',
      error: sentError,
    })
  }
})

api.get('/balance', (req, res) => {
  console.log({
    headers: req.headers,
    body: req.body,
  })

  const { key } = authorize(req)

  if (!key)
    return res.status(401).send()

  try {
    const balance = DataStore.balances.get(key)
    // console.log({
    //     in: '/balance',
    //     balance,
    // })

    return res.status(200).send(balance)
  } catch (err) {
    console.error(err)

    const sentError = {}
    if (err.message)
      sentError.message = err.message
    if (err.detail)
      sentError.detail = err.detail

    return res.status(400).send({
      result: 'error',
      error: sentError,
    })
  }
})

api.get('/positions', (req, res) => {
  const { key } = authorize(req)

  if (!key)
    return res.status(401).send()

  try {
    const positions = DataStore.positions.get(key)

    if (!positions)
      return res.status(404)

    return res.status(200).send(positions)
  } catch (err) {
    console.error(err)
    return res.status(400).send()
  }
})

api.get('/state', (req, res) => {
  const { key } = authorize(req)

  if (!key)
    return res.status(401).send()

  const allowed = ['admin', 'owner', 'test']
  if (!hasOneOfRoles(key, allowed))
    return res.status(403).send()

  try {
    const state = DataStore.getState()
    const { balances } = state
    const { books: _depths, ...noPrices } = state

    return res.status(200).send({
      ...noPrices,
      balances,
    })
  } catch (err) {
    console.error(err)
    return res.status(400).send()
  }
})

api.get('/prices', (req, res) => {
  const { key } = authorize(req)

  if (!key)
    return res.status(401).send()

  const allowed = ['admin', 'owner', 'test']
  if (!hasOneOfRoles(key, allowed))
    return res.status(403).send()

  try {
    const { remora, ...externals } = priceFeed.externals()

    return res.status(200).send({ externals, remora })
  } catch (err) {
    console.error(err)
    return res.status(500).send()
  }
})

api.get('/routing', (req, res) => {
  const { key } = authorize(req)

  if (!key)
    return res.status(401).send()

  const allowed = ['admin', 'owner', 'test']
  if (!hasOneOfRoles(key, allowed))
    return res.status(403).send()

  try {
    const routing = priceFeed.routing()

    return res.status(200).send(routing)
  } catch (err) {
    console.error(err)
    return res.status(500).send()
  }
})

export default api
