using UnityEngine;
using System.Collections.Generic;
using System.Linq;
using Verse.Sound;
using Verse;

namespace RimWorld
{

public struct BulletImpactData
{
    public IntVec3 impactPosition;
    public Bullet bullet;
    public Thing hitThing;
}

public class Bullet : Projectile
{
	protected override void Impact(Thing hitThing)
	{
		var map = Map; // before Impact!
        var position = Position;

		base.Impact(hitThing);
		
		var impact = new BattleLogEntry_RangedImpact(launcher, hitThing, intendedTarget.Thing, equipmentDef, def, targetCoverDef);
		Find.BattleLog.Add(impact);

        NotifyImpact(hitThing, map, position);

		if( hitThing != null )
		{
			var dinfo = new DamageInfo(def.projectile.damageDef, DamageAmount, armorPenetration: ArmorPenetration, angle: ExactRotation.eulerAngles.y, instigator: launcher, weapon: equipmentDef, intendedTarget: intendedTarget.Thing);
			hitThing.TakeDamage(dinfo).AssociateWithLog(impact);

			var hitPawn = hitThing as Pawn;
			if( hitPawn != null && hitPawn.stances != null && hitPawn.BodySize <= def.projectile.StoppingPower + 0.001f )
				hitPawn.stances.StaggerFor(Pawn_StanceTracker.StaggerBulletImpactTicks);

			if (def.projectile.extraDamages != null)
			{
				foreach (var d in def.projectile.extraDamages)
				{
					if (Rand.Chance(d.chance))
					{
						var extraDinfo = new DamageInfo(d.def, d.amount, d.AdjustedArmorPenetration(), ExactRotation.eulerAngles.y, launcher, weapon: equipmentDef, intendedTarget: intendedTarget.Thing);
						hitThing.TakeDamage(extraDinfo).AssociateWithLog(impact);
					}
				}
			}
		}
		else
		{
			SoundDefOf.BulletImpact_Ground.PlayOneShot(new TargetInfo(Position, map));

			if( Position.GetTerrain(map).takeSplashes )
				MoteMaker.MakeWaterSplash(ExactPosition, map, Mathf.Sqrt(DamageAmount) * MoteSplash.SizeGunfire, MoteSplash.VelocityGunfire);
			else
				MoteMaker.MakeStaticMote(ExactPosition, map, ThingDefOf.Mote_ShotHit_Dirt);
        }
    }

    private void NotifyImpact(Thing hitThing, Map map, IntVec3 position)
    {
        var impactNotificationData = new BulletImpactData {bullet = this, hitThing = hitThing, impactPosition = position};

        hitThing?.Notify_BulletImpactNearby(impactNotificationData);

        int cellCount = 9;
        for( int i = 0; i < cellCount; i++ )
        {
            var cell = position + GenRadial.RadialPattern[i];
            if( !cell.InBounds(map) )
                continue;

            var things = cell.GetThingList(map);
            for( int j = 0; j < things.Count; j++ )
            {
                if (things[j] != hitThing)
                    things[j].Notify_BulletImpactNearby(impactNotificationData);
            }
        }
    }
}
}