﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Xml;
using Verse;
using UnityEngine;


namespace RimWorld{
public class PawnGenOption
{
	//Config
	public PawnKindDef			kind;
	public float 				selectionWeight;

	//Properties
	public float Cost{get{return kind.combatPower;}}

	public override string ToString()
	{
		return "(" + (kind!=null?kind.ToString():"null")
			+ " w=" + selectionWeight.ToString("F2")
			+ " c=" + (kind!=null?Cost.ToString("F2"):"null") + ")";
	}

	public void LoadDataFromXmlCustom( XmlNode xmlRoot )
	{
		DirectXmlCrossRefLoader.RegisterObjectWantsCrossRef( this, "kind", xmlRoot.Name );
		selectionWeight = ParseHelper.FromString<float>( xmlRoot.FirstChild.Value );
	}
}

public class RoyalImplantRule
{
	public HediffDef implantHediff;
	public RoyalTitleDef minTitle;
	public int maxLevel;
}

public class FactionDef : Def
{
	//General config
	public bool					isPlayer = false;
	public RulePackDef			factionNameMaker;
	public RulePackDef			settlementNameMaker;
	public RulePackDef			playerInitialSettlementNameMaker;
	[MustTranslate] public string fixedName = null;
	public bool					humanlikeFaction = true;
	public bool					hidden = false;
	public float				listOrderPriority = 0f;
	public List<PawnGroupMaker>	pawnGroupMakers = null;
	public SimpleCurve			raidCommonalityFromPointsCurve = null;
	public bool					autoFlee = true;
	public FloatRange			attackersDownPercentageRangeForAutoFlee = new FloatRange(.4f, .7f);
	public bool					canSiege = false;
	public bool					canStageAttacks = false;
	public bool					canUseAvoidGrid = true;
	public float				earliestRaidDays = 0;
	public FloatRange			allowedArrivalTemperatureRange = new FloatRange(-1000, 1000);
	public PawnKindDef			basicMemberKind;
	public List<ResearchProjectTagDef>	startingResearchTags = null;
	public List<ResearchProjectTagDef>  startingTechprintsResearchTags = null;
	[NoTranslate] public List<string>	recipePrerequisiteTags = null;
	public bool					rescueesCanJoin = false;
	[MustTranslate] public string pawnSingular = "member";
	[MustTranslate] public string pawnsPlural = "members";
	public string				leaderTitle = "leader"; //Not MustTranslate because many faction defs never have leaders
	public string				leaderTitleFemale; //Not MustTranslate because many faction defs never have leaders
    [MustTranslate] public string royalFavorLabel;
    [NoTranslate] public string royalFavorIconPath;
	public List<PawnKindDef>	fixedLeaderKinds = null;
	public bool					leaderForceGenerateNewPawn;
	public float				forageabilityFactor = 1f;
	public SimpleCurve			maxPawnCostPerTotalPointsCurve = null;
	public List<string>			royalTitleTags;
	public string	            categoryTag;
    public bool                 hostileToFactionlessHumanlikes;

	//Faction generation
	public int					requiredCountAtGameStart = 0;
	public int					maxCountAtGameStart = 9999;
	public bool					canMakeRandomly = false;
	public float				settlementGenerationWeight = 0f;
    public bool                 generateNewLeaderFromMapMembersOnly = false;

	//Humanlike faction config
	public RulePackDef			pawnNameMaker;
	public RulePackDef			pawnNameMakerFemale;
	public TechLevel			techLevel = TechLevel.Undefined;
	[NoTranslate] public List<BackstoryCategoryFilter> backstoryFilters = null;
	[NoTranslate] List<string> backstoryCategories = null;
	[NoTranslate] public List<string> hairTags = new List<string>();
	public ThingFilter			apparelStuffFilter = null;
	public ThingSetMakerDef     raidLootMaker = null;
    public SimpleCurve          raidLootValueFromPointsCurve = DefaultRaidLootValueFromPointsCurve_NewTemp;
	public List<TraderKindDef>	caravanTraderKinds = new List<TraderKindDef>();
	public List<TraderKindDef>	visitorTraderKinds = new List<TraderKindDef>();
	public List<TraderKindDef>	baseTraderKinds = new List<TraderKindDef>();
	public float				geneticVariance = 1f;

	//Relations (can apply to non-humanlike factions)
	public IntRange				startingGoodwill = IntRange.zero;
	public bool					mustStartOneEnemy = false;
	public IntRange				naturalColonyGoodwill = IntRange.zero;
	public float				goodwillDailyGain = 0;
	public float				goodwillDailyFall = 0;
	public bool					permanentEnemy = false;
    public bool                 permanentEnemyToEveryoneExceptPlayer = false;
    public List<FactionDef>     permanentEnemyToEveryoneExcept;

	//World drawing
	[NoTranslate] public string	settlementTexturePath;
	[NoTranslate] public string	factionIconPath;
	public List<Color>			colorSpectrum;

	// Royal titles
	public List<PawnRelationDef> royalTitleInheritanceRelations;
	public Type royalTitleInheritanceWorkerClass = null;
	public List<RoyalImplantRule> royalImplantRules;
    [System.Obsolete("Will be removed in the future")]
    public RoyalTitleDef minTitleForBladelinkWeapons;
    public string renounceTitleMessage;

	//Unsaved
	[Unsaved] private Texture2D	factionIcon;
	[Unsaved] private Texture2D	settlementTexture;
    [Unsaved] private Texture2D royalFavorIcon;

    //Cache
	[Unsaved] private List<RoyalTitleDef> royalTitlesAwardableInSeniorityOrderForReading;
	[Unsaved] private List<RoyalTitleDef> royalTitlesAllInSeniorityOrderForReading;
	[Unsaved] private RoyalTitleInheritanceWorker royalTitleInheritanceWorker;

    // NewTemp: For mod compatibility only; remove for 1.2 and add a ConfigError requiring <raidLootValueFromPointsCurve> to be defined
    [System.Obsolete]
    private static readonly SimpleCurve DefaultRaidLootValueFromPointsCurve_NewTemp = new SimpleCurve()
    {
        new CurvePoint(35, 15),
        new CurvePoint(100, 120),
        new CurvePoint(1000, 500),
        new CurvePoint(2000, 800),
        new CurvePoint(4000, 1000)
    };

    
	//Properties
	public List<RoyalTitleDef> RoyalTitlesAwardableInSeniorityOrderForReading
	{
		get
		{
			// Cache init
			if (royalTitlesAwardableInSeniorityOrderForReading == null)
			{
				royalTitlesAwardableInSeniorityOrderForReading = new List<RoyalTitleDef>();
				if (royalTitleTags != null && ModLister.RoyaltyInstalled)
				{
					foreach (var titleDef in DefDatabase<RoyalTitleDef>.AllDefsListForReading)
					{
						if (titleDef.Awardable && titleDef.tags.SharesElementWith(royalTitleTags))
							royalTitlesAwardableInSeniorityOrderForReading.Add(titleDef);
					}

					royalTitlesAwardableInSeniorityOrderForReading.SortBy(t => t.seniority);
				}
			}
			return royalTitlesAwardableInSeniorityOrderForReading;
		}
	}
    public List<RoyalTitleDef> RoyalTitlesAllInSeniorityOrderForReading
    {
        get
        {
			// Cache init
			if (royalTitlesAllInSeniorityOrderForReading == null)
			{
				royalTitlesAllInSeniorityOrderForReading = new List<RoyalTitleDef>();
				if (royalTitleTags != null && ModLister.RoyaltyInstalled)
				{
					foreach (var titleDef in DefDatabase<RoyalTitleDef>.AllDefsListForReading)
					{
						if (titleDef.tags.SharesElementWith(royalTitleTags))
							royalTitlesAllInSeniorityOrderForReading.Add(titleDef);
					}

					royalTitlesAllInSeniorityOrderForReading.SortBy(t => t.seniority);
				}
			}
			return royalTitlesAllInSeniorityOrderForReading;
        }
    }
	public RoyalTitleInheritanceWorker RoyalTitleInheritanceWorker
	{
		get
		{
			if (royalTitleInheritanceWorker == null && royalTitleInheritanceWorkerClass != null)
				royalTitleInheritanceWorker = (RoyalTitleInheritanceWorker)Activator.CreateInstance(royalTitleInheritanceWorkerClass);
			return royalTitleInheritanceWorker;
		}
	}
	public bool CanEverBeNonHostile
	{
		get
		{
			return !permanentEnemy;
		}
	}
	public Texture2D SettlementTexture
	{
		get
		{
			if( settlementTexture == null )
			{
				if( !settlementTexturePath.NullOrEmpty() )
					settlementTexture = ContentFinder<Texture2D>.Get(settlementTexturePath);
				else
					settlementTexture = BaseContent.BadTex;
			}

			return settlementTexture;
		}
	}
	public Texture2D FactionIcon
	{
		get
		{
			if( factionIcon == null )
			{
				if( !factionIconPath.NullOrEmpty() )
					factionIcon = ContentFinder<Texture2D>.Get(factionIconPath);
				else
					factionIcon = BaseContent.BadTex;
			}

			return factionIcon;
		}
	}
    public Texture2D RoyalFavorIcon
    {
        get
        {
            if( royalFavorIcon == null && !royalFavorIconPath.NullOrEmpty() )
                royalFavorIcon = ContentFinder<Texture2D>.Get(royalFavorIconPath);

            return royalFavorIcon;
        }
    }
	public bool HasRoyalTitles
	{
		get { return RoyalTitlesAwardableInSeniorityOrderForReading.Count > 0; }
	}


	public float MinPointsToGeneratePawnGroup(PawnGroupKindDef groupKind)
	{
		if( pawnGroupMakers == null )
			return 0;

		var groups = pawnGroupMakers.Where(x => x.kindDef == groupKind);

		if( !groups.Any() )
			return 0;

		return groups.Min(pgm => pgm.MinPointsToGenerateAnything);
	}

	public bool CanUseStuffForApparel( ThingDef stuffDef )
	{
		if( apparelStuffFilter == null )
			return true;

		return apparelStuffFilter.Allows( stuffDef );
	}

	public float RaidCommonalityFromPoints( float points )
	{
		if( points < 0 || raidCommonalityFromPointsCurve == null )
			return 1f;

		return raidCommonalityFromPointsCurve.Evaluate(points);
	}

	public override void ResolveReferences()
	{
		base.ResolveReferences();

		if( apparelStuffFilter != null )
			apparelStuffFilter.ResolveReferences();
	}
	
	public override void PostLoad()
	{
		if (backstoryCategories != null && backstoryCategories.Count > 0)
		{
			if (backstoryFilters == null)
				backstoryFilters = new List<BackstoryCategoryFilter>();
			backstoryFilters.Add(new BackstoryCategoryFilter() { categories = backstoryCategories });
		}
	}
	
	public override IEnumerable<string> ConfigErrors()
	{
		foreach( var error in base.ConfigErrors() )
			yield return error;

		if( pawnGroupMakers != null && maxPawnCostPerTotalPointsCurve == null )
			yield return "has pawnGroupMakers but missing maxPawnCostPerTotalPointsCurve";

		if( !isPlayer && factionNameMaker == null && fixedName == null )
			yield return "FactionTypeDef " + defName + " lacks a factionNameMaker and a fixedName.";

		if( techLevel == TechLevel.Undefined )
			yield return defName + " has no tech level.";

		if( humanlikeFaction )
		{
			if( backstoryFilters.NullOrEmpty() )
				yield return defName + " is humanlikeFaction but has no backstory categories.";

			if( hairTags.Count == 0 )
				yield return defName + " is humanlikeFaction but has no hairTags.";
		}

		if( isPlayer )
		{
			if( settlementNameMaker == null )
				yield return "isPlayer is true but settlementNameMaker is null";

			if( factionNameMaker == null )
				yield return "isPlayer is true but factionNameMaker is null";
			
			if( playerInitialSettlementNameMaker == null )
				yield return "isPlayer is true but playerInitialSettlementNameMaker is null";
		}

		if( permanentEnemy )
		{
			if( mustStartOneEnemy )
				yield return "permanentEnemy has mustStartOneEnemy = true, which is redundant";

			if( goodwillDailyFall != 0 || goodwillDailyGain != 0 )
				yield return "permanentEnemy has a goodwillDailyFall or goodwillDailyGain";

			if( startingGoodwill != IntRange.zero )
				yield return "permanentEnemy has a startingGoodwill defined";

			if (naturalColonyGoodwill != IntRange.zero )
				yield return "permanentEnemy has a naturalColonyGoodwill defined";
		}
	}

	public static FactionDef Named( string defName )
	{
		return DefDatabase<FactionDef>.GetNamed(defName);
	}
	
	public RulePackDef GetNameMaker( Gender gender )
	{
		if( gender == Gender.Female && pawnNameMakerFemale != null )
			return pawnNameMakerFemale;

		return pawnNameMaker;
	}
}}

